/*
 * Copyright 2001 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.bacnet.config.BBacnetConfigDeviceExt;
import javax.baja.bacnet.config.BBacnetConfigFolder;
import javax.baja.bacnet.datatypes.BBacnetAddress;
import javax.baja.bacnet.datatypes.BBacnetArray;
import javax.baja.bacnet.datatypes.BBacnetListOf;
import javax.baja.bacnet.datatypes.BBacnetObjectIdentifier;
import javax.baja.bacnet.datatypes.BBacnetObjectPropertyReference;
import javax.baja.bacnet.datatypes.BBacnetPriorityValue;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.enums.BExtensibleEnumList;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.PropertyReference;
import javax.baja.bacnet.util.BIBacnetPollable;
import javax.baja.bacnet.util.BacnetBitStringUtil;
import javax.baja.bacnet.util.PollListEntry;
import javax.baja.bacnet.util.PropertyInfo;
import javax.baja.data.BIDataValue;
import javax.baja.driver.loadable.BDownloadParameters;
import javax.baja.driver.loadable.BLoadable;
import javax.baja.driver.loadable.BUploadParameters;
import javax.baja.driver.util.BPollFrequency;
import javax.baja.naming.SlotPath;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.util.Array;
import javax.baja.nre.util.ByteArrayUtil;
import javax.baja.registry.TypeInfo;
import javax.baja.space.BComponentSpace;
import javax.baja.spy.SpyWriter;
import javax.baja.status.BStatus;
import javax.baja.sys.Action;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BDynamicEnum;
import javax.baja.sys.BEnum;
import javax.baja.sys.BEnumRange;
import javax.baja.sys.BFacets;
import javax.baja.sys.BIcon;
import javax.baja.sys.BInteger;
import javax.baja.sys.BObject;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BValue;
import javax.baja.sys.Clock;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.NotRunningException;
import javax.baja.sys.Property;
import javax.baja.sys.Slot;
import javax.baja.sys.SlotCursor;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.BTypeSpec;
import javax.baja.util.Lexicon;

import com.tridium.bacnet.BacUtil;
import com.tridium.bacnet.asn.AsnConst;
import com.tridium.bacnet.asn.AsnInputStream;
import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NBacnetPropertyReference;
import com.tridium.bacnet.asn.NReadAccessResult;
import com.tridium.bacnet.asn.NReadAccessSpec;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.asn.NWriteAccessSpec;
import com.tridium.bacnet.stack.BBacnetPoll;
import com.tridium.bacnet.stack.BBacnetStack;
import com.tridium.bacnet.stack.client.BBacnetClientLayer;
import com.tridium.bacnet.stack.transport.TransactionException;

/**
 * @author Craig Gemmill
 * @version $Revision: 17$ $Date: 12/17/01 9:14:09 AM$
 * @creation 20 Jul 00
 * @since Niagara 3 Bacnet 1.0
 */
@NiagaraType
@NiagaraProperty(
  name = "pollFrequency",
  type = "BPollFrequency",
  defaultValue = "BPollFrequency.normal"
)
@NiagaraProperty(
  name = "status",
  type = "BStatus",
  defaultValue = "BStatus.ok",
  flags = Flags.TRANSIENT | Flags.READONLY
)
@NiagaraProperty(
  name = "faultCause",
  type = "String",
  defaultValue = "",
  flags = Flags.TRANSIENT | Flags.READONLY
)
@NiagaraProperty(
  name = "objectId",
  type = "BBacnetObjectIdentifier",
  defaultValue = "BBacnetObjectIdentifier.DEFAULT",
  flags = Flags.SUMMARY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.OBJECT_IDENTIFIER, ASN_OBJECT_IDENTIFIER)")
)
@NiagaraProperty(
  name = "objectName",
  type = "String",
  defaultValue = "",
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.OBJECT_NAME, ASN_CHARACTER_STRING)")
)
@NiagaraProperty(
  name = "objectType",
  type = "BEnum",
  defaultValue = "BDynamicEnum.make(0, BEnumRange.make(BBacnetObjectType.TYPE))",
  flags = Flags.READONLY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.OBJECT_TYPE, ASN_ENUMERATED)")
)
@NiagaraAction(
  name = "download",
  parameterType = "BDownloadParameters",
  defaultValue = "new BDownloadParameters()",
  flags = Flags.ASYNC | Flags.HIDDEN,
  override = true
)
@NiagaraAction(
  name = "readBacnetProperty",
  parameterType = "BEnum",
  defaultValue = "BDynamicEnum.make(BBacnetPropertyIdentifier.presentValue)",
  returnType = "BValue",
  flags = Flags.HIDDEN
)
@NiagaraAction(
  name = "writeBacnetProperty",
  parameterType = "BEnum",
  defaultValue = "BDynamicEnum.make(BBacnetPropertyIdentifier.presentValue)",
  flags = Flags.HIDDEN
)
@NiagaraAction(
  name = "uploadRequiredProperties",
  flags = Flags.HIDDEN
)
@NiagaraAction(
  name = "uploadOptionalProperties",
  flags = Flags.HIDDEN
)
public class BBacnetObject
  extends BLoadable
  implements BacnetConst,
             BIBacnetPollable
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.BBacnetObject(1117206849)1.0$ @*/
/* Generated Thu Jun 02 14:30:03 EDT 2022 by Slot-o-Matic (c) Tridium, Inc. 2012-2022 */

  //region Property "pollFrequency"

  /**
   * Slot for the {@code pollFrequency} property.
   * @see #getPollFrequency
   * @see #setPollFrequency
   */
  @Generated
  public static final Property pollFrequency = newProperty(0, BPollFrequency.normal, null);

  /**
   * Get the {@code pollFrequency} property.
   * @see #pollFrequency
   */
  @Generated
  public BPollFrequency getPollFrequency() { return (BPollFrequency)get(pollFrequency); }

  /**
   * Set the {@code pollFrequency} property.
   * @see #pollFrequency
   */
  @Generated
  public void setPollFrequency(BPollFrequency v) { set(pollFrequency, v, null); }

  //endregion Property "pollFrequency"

  //region Property "status"

  /**
   * Slot for the {@code status} property.
   * @see #getStatus
   * @see #setStatus
   */
  @Generated
  public static final Property status = newProperty(Flags.TRANSIENT | Flags.READONLY, BStatus.ok, null);

  /**
   * Get the {@code status} property.
   * @see #status
   */
  @Generated
  public BStatus getStatus() { return (BStatus)get(status); }

  /**
   * Set the {@code status} property.
   * @see #status
   */
  @Generated
  public void setStatus(BStatus v) { set(status, v, null); }

  //endregion Property "status"

  //region Property "faultCause"

  /**
   * Slot for the {@code faultCause} property.
   * @see #getFaultCause
   * @see #setFaultCause
   */
  @Generated
  public static final Property faultCause = newProperty(Flags.TRANSIENT | Flags.READONLY, "", null);

  /**
   * Get the {@code faultCause} property.
   * @see #faultCause
   */
  @Generated
  public String getFaultCause() { return getString(faultCause); }

  /**
   * Set the {@code faultCause} property.
   * @see #faultCause
   */
  @Generated
  public void setFaultCause(String v) { setString(faultCause, v, null); }

  //endregion Property "faultCause"

  //region Property "objectId"

  /**
   * Slot for the {@code objectId} property.
   * @see #getObjectId
   * @see #setObjectId
   */
  @Generated
  public static final Property objectId = newProperty(Flags.SUMMARY, BBacnetObjectIdentifier.DEFAULT, makeFacets(BBacnetPropertyIdentifier.OBJECT_IDENTIFIER, ASN_OBJECT_IDENTIFIER));

  /**
   * Get the {@code objectId} property.
   * @see #objectId
   */
  @Generated
  public BBacnetObjectIdentifier getObjectId() { return (BBacnetObjectIdentifier)get(objectId); }

  /**
   * Set the {@code objectId} property.
   * @see #objectId
   */
  @Generated
  public void setObjectId(BBacnetObjectIdentifier v) { set(objectId, v, null); }

  //endregion Property "objectId"

  //region Property "objectName"

  /**
   * Slot for the {@code objectName} property.
   * @see #getObjectName
   * @see #setObjectName
   */
  @Generated
  public static final Property objectName = newProperty(0, "", makeFacets(BBacnetPropertyIdentifier.OBJECT_NAME, ASN_CHARACTER_STRING));

  /**
   * Get the {@code objectName} property.
   * @see #objectName
   */
  @Generated
  public String getObjectName() { return getString(objectName); }

  /**
   * Set the {@code objectName} property.
   * @see #objectName
   */
  @Generated
  public void setObjectName(String v) { setString(objectName, v, null); }

  //endregion Property "objectName"

  //region Property "objectType"

  /**
   * Slot for the {@code objectType} property.
   * @see #getObjectType
   * @see #setObjectType
   */
  @Generated
  public static final Property objectType = newProperty(Flags.READONLY, BDynamicEnum.make(0, BEnumRange.make(BBacnetObjectType.TYPE)), makeFacets(BBacnetPropertyIdentifier.OBJECT_TYPE, ASN_ENUMERATED));

  /**
   * Get the {@code objectType} property.
   * @see #objectType
   */
  @Generated
  public BEnum getObjectType() { return (BEnum)get(objectType); }

  /**
   * Set the {@code objectType} property.
   * @see #objectType
   */
  @Generated
  public void setObjectType(BEnum v) { set(objectType, v, null); }

  //endregion Property "objectType"

  //region Action "download"

  /**
   * Slot for the {@code download} action.
   * @see #download(BDownloadParameters parameter)
   */
  @Generated
  public static final Action download = newAction(Flags.ASYNC | Flags.HIDDEN, new BDownloadParameters(), null);

  //endregion Action "download"

  //region Action "readBacnetProperty"

  /**
   * Slot for the {@code readBacnetProperty} action.
   * @see #readBacnetProperty(BEnum parameter)
   */
  @Generated
  public static final Action readBacnetProperty = newAction(Flags.HIDDEN, BDynamicEnum.make(BBacnetPropertyIdentifier.presentValue), null);

  /**
   * Invoke the {@code readBacnetProperty} action.
   * @see #readBacnetProperty
   */
  @Generated
  public BValue readBacnetProperty(BEnum parameter) { return invoke(readBacnetProperty, parameter, null); }

  //endregion Action "readBacnetProperty"

  //region Action "writeBacnetProperty"

  /**
   * Slot for the {@code writeBacnetProperty} action.
   * @see #writeBacnetProperty(BEnum parameter)
   */
  @Generated
  public static final Action writeBacnetProperty = newAction(Flags.HIDDEN, BDynamicEnum.make(BBacnetPropertyIdentifier.presentValue), null);

  /**
   * Invoke the {@code writeBacnetProperty} action.
   * @see #writeBacnetProperty
   */
  @Generated
  public void writeBacnetProperty(BEnum parameter) { invoke(writeBacnetProperty, parameter, null); }

  //endregion Action "writeBacnetProperty"

  //region Action "uploadRequiredProperties"

  /**
   * Slot for the {@code uploadRequiredProperties} action.
   * @see #uploadRequiredProperties()
   */
  @Generated
  public static final Action uploadRequiredProperties = newAction(Flags.HIDDEN, null);

  /**
   * Invoke the {@code uploadRequiredProperties} action.
   * @see #uploadRequiredProperties
   */
  @Generated
  public void uploadRequiredProperties() { invoke(uploadRequiredProperties, null, null); }

  //endregion Action "uploadRequiredProperties"

  //region Action "uploadOptionalProperties"

  /**
   * Slot for the {@code uploadOptionalProperties} action.
   * @see #uploadOptionalProperties()
   */
  @Generated
  public static final Action uploadOptionalProperties = newAction(Flags.HIDDEN, null);

  /**
   * Invoke the {@code uploadOptionalProperties} action.
   * @see #uploadOptionalProperties
   */
  @Generated
  public void uploadOptionalProperties() { invoke(uploadOptionalProperties, null, null); }

  //endregion Action "uploadOptionalProperties"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetObject.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  /**
   * Default constructor.
   */
  public BBacnetObject()
  {
  }

  /**
   * Create a new BBacnetObject from the given object-identifier.
   *
   * @param id the object-identifier specifying the new object.
   * @return a BBacnetObject with the given objectId.
   */
  public static BBacnetObject make(BBacnetObjectIdentifier id)
  {
    if (!initialized) init();
    Array<TypeInfo> o = byObjectType.get(id.getObjectType());
    if (o != null && o.size() > 0)
    {
      TypeInfo element = o.get(0);
      BBacnetObject bo = (BBacnetObject)element.getInstance();
      bo.setObjectId(id);
      return bo;
    }
    return new BBacnetObject();
  }

  /**
   * Get the TypeInfo for the given BACnet Object ID.
   *
   * @deprecated Use getTypeInfos(BBacnetObjectIdentifier) instead
   */
  @Deprecated
  public static TypeInfo getTypeInfo(BBacnetObjectIdentifier id)
  {
    if (!initialized) init();
    Array<TypeInfo> a = byObjectType.get(id.getObjectType());
    if (a != null) return a.first();
    return BBacnetObject.TYPE.getTypeInfo();
  }

  /**
   * Get the TypeInfos available for the given BACnet Object ID.
   */
  public static TypeInfo[] getTypeInfos(BBacnetObjectIdentifier id)
  {
    if (!initialized) init();
    Array<TypeInfo> a = byObjectType.get(id.getObjectType());
    if (a != null) return a.trim();
    return new TypeInfo[] { BBacnetObject.TYPE.getTypeInfo() };
  }


////////////////////////////////////////////////////////////////
//  BComponent Overrides
////////////////////////////////////////////////////////////////

  /**
   * Started.
   */
  public void started()
    throws Exception
  {
    checkConfig();
    buildPolledProperties();
    BBacnetObject obj = config().lookupBacnetObject(getObjectId());
    if ((obj != null) && (obj != this))
    {
      log.severe("Duplicate Bacnet Object ID for config object " + this + " in " + device()
        + "; defaulting objectId!");
      setObjectId(BBacnetObjectIdentifier.make(getObjectType().getOrdinal()));
    }
  }

  /**
   * Stopped.
   */
  public void stopped()
    throws Exception
  {
    try
    {
      BBacnetNetwork network = network();
      if (network != null)
      {
        network.getPollService(this).unsubscribe(this);
      }
    }
    catch (NotRunningException e)
    {
      log.warning("BBacnetObject.stopped:NotRunningException unsubscribing from polling on " + this);
    }
    polledProperties = null;
  }

  /**
   * Property changed.
   */
  public void changed(Property p, Context cx)
  {
    super.changed(p, cx);
    if (!isRunning() || cx == noWrite)
    {
      return;
    }

    if (p.equals(objectId))
    {
      removeAll(null);
      upload(new BUploadParameters());
      if (isSubscribed())
      {
        BBacnetNetwork network = network();
        if (network != null)
        {
          BBacnetPoll pollService = (BBacnetPoll)network.getPollService(this);
          pollService.unsubscribe(this);
          buildPolledProperties();
          pollService.subscribe(this);
        }
      }
      return;
    }

    if (p.equals(pollFrequency))
    {
      if (isSubscribed())
      {
        BBacnetNetwork network = network();
        if (network != null)
        {
          BBacnetPoll pollService = (BBacnetPoll)network.getPollService(this);
          pollService.unsubscribe(this);
          pollService.subscribe(this);
        }
      }
    }

    if (!Flags.isReadonly(this, p) && p.getFacets().getFacet(PID) != null)
    {
      // Schedule all property writes in case the property changed results from some but not all
      // set operations of struct property value.
      BacnetPropertyData propData = getPropertyData(p);
      if (propData != NOT_BACNET_PROPERTY)
      {
        synchronized (writeProps)
        {
          PropertyWrite propWrite = new PropertyWrite(propData, p);
          writeProps.put(propWrite, propWrite);
          if (writePropsTicket == null)
          {
            writePropsTicket = Clock.schedule(this, WRITE_PROPS_DELAY, download, new BDownloadParameters());
          }
        }
      }
    }
  }

  /**
   * For objectId, get the facets from the device's object type facets.
   */
  public BFacets getSlotFacets(Slot slot)
  {
    if (slot.equals(objectId))
    {
      if (!isMounted()) return super.getSlotFacets(slot);
      BFacets f = BBacnetObjectType.getObjectIdFacets(getObjectType().getOrdinal());
      if (f != null)
        return f;
      BBacnetDevice dev = device();//(BBacnetDevice)getDevice();
      if (dev != null)
      {
        BExtensibleEnumList elist = dev.getEnumerationList();
        if (elist != null)
        {
          return elist.getObjectTypeFacets();
        }
      }
    }
    if (slot.equals(objectType))
    {
      BBacnetDevice dev = (BBacnetDevice)getDevice();
      if (dev != null)
      {
        BExtensibleEnumList elist = dev.getEnumerationList();
        if (elist != null)
        {
          return elist.getObjectTypeFacets();
        }
      }
    }

    // FIXX: temp handling of dynamic bit strings.
    if (slot.getName().equals(BBacnetPropertyIdentifier.statusFlags.getTag()))
      return BacnetBitStringUtil.BACNET_STATUS_FLAGS_FACETS;
    if (slot.getName().equals(BBacnetPropertyIdentifier.eventEnable.getTag()))
      return BacnetBitStringUtil.BACNET_EVENT_TRANSITION_BITS_FACETS;
    if (slot.getName().equals(BBacnetPropertyIdentifier.ackedTransitions.getTag()))
      return BacnetBitStringUtil.BACNET_EVENT_TRANSITION_BITS_FACETS;
    if (slot.getName().equals(BBacnetPropertyIdentifier.limitEnable.getTag()))
      return BacnetBitStringUtil.BACNET_LIMIT_ENABLE_FACETS;
    return super.getSlotFacets(slot);
  }

  /**
   * Callback when the component enters the subscribed state.
   */
  public void subscribed()
  {
    if (!isRunning())
    {
      return;
    }

    BBacnetNetwork network = network();
    if (network != null)
    {
      network.getPollService(this).subscribe(this);
    }
    upload(new BUploadParameters(false));
  }

  /**
   * Callback when the component exits the subscribed state.
   */
  public void unsubscribed()
  {
    if (!isRunning())
    {
      return;
    }

    BBacnetNetwork network = network();
    if (network != null)
    {
      network.getPollService(this).unsubscribe(this);
    }
  }

////////////////////////////////////////////////////////////////
// Actions
////////////////////////////////////////////////////////////////

  /**
   * Callback for processing upload on async thread.
   * Default implementation is to call asyncUpload on all
   * children implementing the Loadable interface.
   */
  @SuppressWarnings("unchecked")
  public void doUpload(BUploadParameters p, Context cx)
  {
    // Bail if device is down or disabled, or objectId is bad.
    BBacnetDevice device = device();
    if (device == null || !device.getEnabled() || device.getStatus().isDown())
    {
      if (log.isLoggable(Level.FINE))
      {
        log.fine((device != null ? device.getName() : "null") + " is either disabled or status is down, object upload is unsuccessful.");
      }
      return;
    }

    if (!getObjectId().isValid())
    {
      return;
    }

    setStatus(BStatus.make(getStatus().getBits() | BStatus.STALE, BFacets.make("upload", "PENDING")));

    // If the device does not support ReadPropertyMultiple, we can only
    // read the possible properties, and see what exists.
    if (!device.isServiceSupported("readPropertyMultiple"))
    {
      uploadIndividual(device, new NReadAccessSpec(getObjectId(), getPropertyIds(device)));
    }
    else
    {
      // First, try to read all implemented properties using the "all" propertyId.
      @SuppressWarnings("rawtypes") Vector specs = new Vector();
      specs.add(new NReadAccessSpec(getObjectId(), BBacnetPropertyIdentifier.ALL));
      @SuppressWarnings("rawtypes") Vector vals = null;
      boolean ok = false;
      try
      {
        vals = client().readPropertyMultiple(device.getAddress(), specs);

        // If vals is null, comm is disabled so just quit.
        if (vals == null) return;

        @SuppressWarnings("rawtypes") Iterator it = ((NReadAccessResult)vals.elementAt(0)).getResults();
        updateProperties(device, it);
        ok = true;
      }
      catch (Exception e)
      {
        if (log.isLoggable(Level.FINE))
        {
          log.fine("Exception uploading " + this + " using rpm(ALL):" + e);
        }
      }

      if (!ok)
      {
        try
        {
          // Get required, then optional, properties.
          specs.clear();
          specs.add(new NReadAccessSpec(getObjectId(), BBacnetPropertyIdentifier.REQUIRED));
          vals = client().readPropertyMultiple(device.getAddress(), specs);
          @SuppressWarnings("rawtypes") Iterator it = ((NReadAccessResult)vals.elementAt(0)).getResults();
          updateProperties(device, it);
          specs.clear();

          specs.add(new NReadAccessSpec(getObjectId(), BBacnetPropertyIdentifier.OPTIONAL));
          vals = client().readPropertyMultiple(device.getAddress(), specs);
          it = ((NReadAccessResult)vals.elementAt(0)).getResults();
          updateProperties(device, it);

          ok = true;
        }
        catch (Exception e)
        {
          if (log.isLoggable(Level.FINE))
          {
            log.fine("Exception uploading " + this + " using rpm(REQ/OPT):" + e);
          }
        }
      }

      if (!ok)
      {
        uploadIndividual(device, new NReadAccessSpec(getObjectId(), getPropertyIds(device)));
      }
    }

    // Now that all values have been read from the device, set the
    // facets describing the output property, and update.
    setOutputFacets();
    BComponentSpace space = getComponentSpace();
    if (space != null) space.update(this, 0);
    buildPolledProperties();
    setStatus(BStatus.ok);
    if (log.isLoggable(Level.FINEST))
    {
      log.finest(device.getName() + " object upload execution finish.");
    }
  }

  private int[] getPropertyIds(BBacnetDevice device)
  {
    try
    {
      return readPropertyList(device);
    }
    catch (Exception e)
    {
      if (log.isLoggable(Level.FINE))
      {
        log.log(Level.FINE, "Exception reading property list for " + this + ": " + e, e);
      }
    }

    int[] possibleProperties = device.getPossibleProperties(getObjectId());
    // Do not include the Property List property. It would not be returned by a readPropertyMultiple
    // with the "All" or "Required" identifiers.
    return removePropertyListProperty(possibleProperties);
  }

  private int[] removePropertyListProperty(int[] propIds)
  {
    for (int i = 0; i < propIds.length; i++)
    {
      if (propIds[i] == BBacnetPropertyIdentifier.PROPERTY_LIST)
      {
        int[] newPropIds = new int[propIds.length - 1];
        //               src,     srcPos, dest,       destPos, length
        // Copy everything up to the Property List id
        System.arraycopy(propIds, 0,      newPropIds, 0,       i);
        // Copy everything after the Property List id
        System.arraycopy(propIds, i + 1,  newPropIds, i,       propIds.length - i - 1);
        return newPropIds;
      }
    }
    return propIds;
  }

  private int[] readPropertyList(BBacnetDevice device)
    throws BacnetException
  {
    // Proprietary properties are added as BDynamicEnum and not BBacnetPropertyIdentifier
    BBacnetArray propertyList = new BBacnetArray(BDynamicEnum.TYPE);
    readArray(device, BBacnetPropertyIdentifier.PROPERTY_LIST, propertyList);

    BDynamicEnum[] propertyIds = propertyList.getChildren(BDynamicEnum.class);
    // The Property List value will not include the Object Name, Object Type, Object Identifier,
    // and Property List properties. Add the first three properties before adding the rest.
    int[] ordinals = new int[propertyIds.length + 3];
    ordinals[0] = BBacnetPropertyIdentifier.OBJECT_IDENTIFIER;
    ordinals[1] = BBacnetPropertyIdentifier.OBJECT_NAME;
    ordinals[2] = BBacnetPropertyIdentifier.OBJECT_TYPE;
    for (int i = 0; i < propertyIds.length; i++)
    {
      ordinals[i + 3] = propertyIds[i].getOrdinal();
    }
    return ordinals;
  }

  public void doUploadRequiredProperties()
  {
    uploadProperties(BBacnetPropertyIdentifier.required);
  }

  public void doUploadOptionalProperties()
  {
    uploadProperties(BBacnetPropertyIdentifier.optional);
  }
  
  @SuppressWarnings("unchecked")
  private void uploadProperties(final BBacnetPropertyIdentifier propertyId)
  {
    if (propertyId == null)
    {
      if (log.isLoggable(Level.FINE))
      {
        log.fine(lex.get("object.upload.unknown.error"));
      }
      return;
    }

    BBacnetNetwork network = BBacnetNetwork.bacnet();
    BBacnetDevice device = device();
    if (network == null || device == null)
    {
      return;
    }

    network.getWorker().post(() -> {
      try
      {
        @SuppressWarnings("rawtypes") Vector specs = new Vector();
        specs.add(new NReadAccessSpec(getObjectId(), propertyId.getOrdinal()));
        @SuppressWarnings("rawtypes") Vector vals = client().readPropertyMultiple(device.getAddress(), specs);
        @SuppressWarnings("rawtypes") Iterator it = ((NReadAccessResult)vals.elementAt(0)).getResults();
        BBacnetObject.this.updateProperties(device, it);
      }
      catch (BacnetException e)
      {
        if (log.isLoggable(Level.FINE))
        {
          log.log(Level.FINE, lex.getText("object.upload." + propertyId.getTag() + ".error"), e);
        }
      }
    });
  }

  /**
   * Callback for processing download on async thread.
   * Default implementation is to call asyncDownload on all
   * children implementing the  Loadable interface.
   */
  @SuppressWarnings("unchecked")
  public void doDownload(BDownloadParameters p, Context cx)
  {
    List<PropertyWrite> propWrites;
    synchronized (writeProps)
    {
      if (writePropsTicket != null)
      {
        writePropsTicket.cancel();
      }
      writePropsTicket = null;
      propWrites = new ArrayList<>(writeProps.values());
      writeProps.clear();
    }

    BBacnetDevice device = device();
    if (device == null)
    {
      return;
    }

    Iterator<PropertyWrite> iterator = propWrites.iterator();
    while (iterator.hasNext())
    {
      PropertyWrite propWrite = iterator.next();

      // Do not include certain non-writable properties like objectId, objectType, etc.
      int propId = propWrite.propData.propertyId;
      if (propId == BBacnetPropertyIdentifier.OBJECT_IDENTIFIER ||
          propId == BBacnetPropertyIdentifier.OBJECT_TYPE)
      {
        iterator.remove();
      }
    }

    boolean wpmOk = false;
    int firstFailPropId = NOT_USED;
    if (propWrites.size() > 1 && device.isServiceSupported("writePropertyMultiple"))
    {
      try
      {
        NWriteAccessSpec writeAccessSpec = new NWriteAccessSpec(getObjectId());
        for (PropertyWrite propWrite : propWrites)
        {
          if (propWrite.propValue == null)
          {
            propWrite.propValue = toEncodedValue(propWrite.propData, propWrite.configProp);
          }

          int propId = propWrite.propData.propertyId;
          if (propId == BBacnetPropertyIdentifier.PRIORITY_ARRAY)
          {
            // Writes to the priority-array property are redirected to the
            // present-value property of the object, with the priority level
            // equal to the specified index of the array.
            propWrite.propValue = BBacnetClientLayer.handlePriorityArrayOptionalDateTime(propWrite.propValue);
            writeAccessSpec.addPropertyValue(
              BBacnetPropertyIdentifier.PRESENT_VALUE,
              /* propertyArrayIndex */ BacnetConst.NOT_USED,
              propWrite.propValue,
              propWrite.arrayIndex);
          }
          else
          {
            writeAccessSpec.addPropertyValue(propId, propWrite.arrayIndex, propWrite.propValue);
          }
        }

        @SuppressWarnings("rawtypes") Vector writeSpecs = new Vector();
        writeSpecs.add(writeAccessSpec);
        client().writePropertyMultiple(device.getAddress(), writeSpecs);
        wpmOk = true;
      }
      catch (ErrorException e)
      {
        firstFailPropId = ((BBacnetObjectPropertyReference)e.getErrorParameters()[0]).getPropertyId();
        log.info("BACnet Error downloading " + this + ":\nFailed write for " + Arrays.toString(e.getErrorParameters()) + ":" + e);
      }
      catch (BacnetException e)
      {
        log.log(
          Level.INFO,
          "BacnetException downloading " + this + ":" + e,
          log.isLoggable(Level.FINE) ? e : null);
      }
    }

    if (!wpmOk)
    {
      // If the write property multiple service is not supported or fails entirely, attempt to use
      // single write property requests.
      boolean preFailure = firstFailPropId != NOT_USED;
      for (PropertyWrite propWrite : propWrites)
      {
        if (preFailure)
        {
          if (firstFailPropId == propWrite.propData.propertyId)
            preFailure = false;
          else
            continue;
        }

        try
        {
          if (propWrite.propValue == null)
          {
            propWrite.propValue = toEncodedValue(propWrite.propData, propWrite.configProp);
          }

          // Handling of writes to the priority-array property are handled in this call to
          // writeProperty.
          client().writeProperty(
            device.getAddress(),
            getObjectId(),
            propWrite.propData.propertyId,
            propWrite.arrayIndex,
            propWrite.propValue);
        }
        catch (Exception e2)
        {
          log.warning("Cannot write property " + propWrite.configProp + " in " + this + ":" + e2);
        }
      }
    }

    if (log.isLoggable(Level.FINEST))
    {
      log.finest(device.getName() + " object download execution finish.");
    }
  }

  /**
   * Read a property.
   */
  public BValue doReadBacnetProperty(BEnum propId)
    throws BacnetException
  {
    BBacnetDevice device = device();
    if (device != null && !device.isDown())
    {
      Property prop = lookupBacnetProperty(propId.getOrdinal());
      if (prop != null)
      {
        readProperty(prop);
        return get(prop);
      }
    }
    return null;
  }

  /**
   * Write a property.
   */
  public void doWriteBacnetProperty(BEnum propId)
    throws BacnetException
  {
    BBacnetDevice device = device();
    if (device != null && !device.isDown())
    {
      Property prop = lookupBacnetProperty(propId.getOrdinal());
      if (prop != null)
      {
        writeProperty(prop);
      }
    }
  }

////////////////////////////////////////////////////////////////
// Convenience Access
////////////////////////////////////////////////////////////////

  /**
   * @return the BBacnetNetwork containing this BBacnetObject.
   */
  protected final BBacnetNetwork network()
  {
    return (config != null) ? config.network() : null;
  }

  /**
   * @return the BBacnetDevice containing this BBacnetObject.
   */
  public final BBacnetDevice device()
  {
    if (config != null)
    {
      return config.device();
    }
    BComplex parent = getParent();
    while (parent != null)
    {
      if (parent instanceof BBacnetDevice)
      {
        return (BBacnetDevice)parent;
      }
      parent = parent.getParent();
    }
    return null;
  }

  /**
   * @return the BBacnetDevice containing this BBacnetObject.
   */
  protected final BBacnetConfigDeviceExt config()
  {
    return config;
  }

  private static BBacnetClientLayer client()
  {
    return ((BBacnetStack)BBacnetNetwork.bacnet().getBacnetComm()).getClient();
  }

  /**
   * To String.
   */
  public String toString(Context context)
  {
    return getName() + " [" + getObjectId().toString(context) + "]";
  }

  /**
   * Subclasses that have a present value property should
   * override this method and return this property.  The
   * default returns null.
   */
  public Property getPresentValueProperty()
  {
    return null;
  }

  /**
   * Set the output facets.
   * Object types with properties containing meta-data about the main value
   * can use this to set a slot containing these facets.
   */
  protected void setOutputFacets()
  {
  }

  /**
   * Should this property ID be polled?
   * Override point for objects to filter properties for polling, e.g.,
   * Object_List in Device object, or Log_Buffer in Trend Log.
   */
  protected boolean shouldPoll(int propertyId)
  {
    return true;
  }

  /**
   * Convert the property to an ASN.1-encoded byte array.
   * Subclasses with properties requiring specialized encoding
   * may need to override this method.
   *
   * @param d
   * @param p
   * @return encoded byte array
   */
  protected byte[] toEncodedValue(BacnetPropertyData d, Property p)
  {
    return AsnUtil.toAsn(d.getAsnType(), get(p));
  }

////////////////////////////////////////////////////////////////
// Overrides
////////////////////////////////////////////////////////////////

  @Override
  public boolean isParentLegal(BComponent parent)
  {
    if (parent instanceof BBacnetConfigFolder && (parent.getParent() instanceof BBacnetConfigFolder)) // Not to support nested folders.
    {
      return false;
    }
    return super.isParentLegal(parent);
  }

////////////////////////////////////////////////////////////////
// Bacnet Specification
////////////////////////////////////////////////////////////////

  /**
   * Read a Bacnet property.
   *
   * @param prop
   */
  public void readProperty(Property prop)
    throws BacnetException
  {
    BBacnetDevice device = device();
    BacnetPropertyData d = getPropertyData(prop);
    if (d == NOT_BACNET_PROPERTY || device == null || device.isDown())
    {
      return/* false*/;
    }

    // Read array properties differently - they can fall back to a
    // one-at-a-time approach if segmentation constraints require.
    if (prop.getType() == BBacnetArray.TYPE)
    {
      readArray(device, d.getPropertyId(), (BBacnetArray) get(prop));
    }
    else
    {
      byte[] encodedValue = null;
      encodedValue = client().readProperty(device.getAddress(),
        getObjectId(),
        d.getPropertyId());
      set(prop, AsnUtil.fromAsn(d.getAsnType(), encodedValue, get(prop)), noWrite);
    }
  }

/** Not Yet...
 public void readPropertyMultiple(Property[] props)
 {
 if (props == null) return;
 Vector refs = new Vector();
 for (int i=0; i<props.length; i++)
 {
 BInteger pId = (BInteger)props[i].getFacets().getFacet(PID);
 if (pId != null)
 refs.addPropertyReference(pId.getInt());
 }
 Vector results = client().readPropertyMultiple(device().getAddress(),
 getObjectId(),
 refs);
 for (int i=0; i<props.length; i++)
 {

 }
 }
 */
  /**
   * Write a Bacnet property.
   *
   * @param prop
   */
  public void writeProperty(Property prop)
    throws BacnetException
  {
    BBacnetDevice device = device();
    BacnetPropertyData d = getPropertyData(prop);
    if (d == NOT_BACNET_PROPERTY || device == null)
    {
      return;
    }

    if (!device.isServiceSupported("writeProperty"))
    {
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.writeProperty"));
    }

    client().writeProperty(device.getAddress(),
      getObjectId(),
      d.getPropertyId(),
      NOT_USED,
      toEncodedValue(d, prop),
      NOT_USED);
  }

  /**
   * Write a Bacnet property.
   *
   * @param prop
   * @param arrayIndex
   * @param encodedValue
   */
  public void writeProperty(Property prop,
                            int arrayIndex,
                            byte[] encodedValue)
    throws BacnetException
  {
    BBacnetDevice device = device();
    BacnetPropertyData propData = getPropertyData(prop);
    if (propData == NOT_BACNET_PROPERTY || device == null)
    {
      return;
    }

    if (!device.isServiceSupported("writeProperty"))
    {
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.writeProperty"));
    }

    synchronized (writeProps)
    {
      PropertyWrite propWrite = new PropertyWrite(propData, arrayIndex, encodedValue);
      writeProps.put(propWrite, propWrite);
      if (writePropsTicket == null)
      {
        writePropsTicket = Clock.schedule(this, WRITE_PROPS_DELAY, download, new BDownloadParameters());
      }
    }
  }

  /**
   * Add an element to a list property.
   *
   * @param prop
   * @param listElement
   */
  public void addListElement(Property prop,
                             BValue listElement)
    throws BacnetException
  {
    // Make sure this is a BACnet ListOf property.
    BBacnetDevice device = device();
    BacnetPropertyData d = getPropertyData(prop);
    if (d == NOT_BACNET_PROPERTY || device == null)
    {
      return;
    }

    if (!get(prop).getType().is(BBacnetListOf.TYPE))
    {
      return;
    }

    if (!device.isServiceSupported("addListElement"))
    {
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.addListElement"));
    }

    byte[] encodedListElement = AsnUtil.toAsn(listElement);

    if(getObjectId().getInstanceNumber() != -1)
    {
      client().addListElement(device.getAddress(),
        getObjectId(),
        d.getPropertyId(),
        NOT_USED,
        encodedListElement);
    }
  }

  /**
   * Remove an element from a list property.
   *
   * @param prop
   * @param listElement
   */
  public void removeListElement(Property prop,
                                BValue listElement)
    throws BacnetException
  {
    // Make sure this is a BACnet ListOf property.
    BBacnetDevice device = device();
    BacnetPropertyData d = getPropertyData(prop);
    if (d == NOT_BACNET_PROPERTY || device == null)
    {
      return;
    }

    if (!get(prop).getType().is(BBacnetListOf.TYPE))
    {
      return;
    }

    if (!device.isServiceSupported("writeProperty"))
    {
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.removeListElement"));
    }

    byte[] encodedListElement = AsnUtil.toAsn(listElement);
    client().removeListElement(device.getAddress(),
      getObjectId(),
      d.getPropertyId(),
      NOT_USED,
      encodedListElement);
  }

////////////////////////////////////////////////////////////////
// BIBacnetPollable
////////////////////////////////////////////////////////////////

  /**
   * Get the pollable type of this object.
   *
   * @return one of the pollable types defined in BIBacnetPollable.
   */
  public final int getPollableType()
  {
    return BACNET_POLLABLE_OBJECT;
  }

  /**
   * Get the poll frequency.
   * @return the poll frequency for this object.
  public final BPollFrequency getPollFrequency()  { return BPollFrequency.normal; }
   */

  /**
   * Poll the object.
   *
   * @return true if the object was successfully polled, false if not.
   * @deprecated
   */
  @Deprecated
  public final boolean poll()
  {
    log.warning("BBacnetObject.poll() is DEPRECATED!!!");
    return false;
  }

  // FIXX:Temporary place holders....

  /**
   * Indicate successful poll.
   */
  public final void readOk()
  {
    setStatus(BStatus.makeFault(getStatus(), false));
    setFaultCause("");
  }

  /**
   * Indicate a failure polling this object.
   *
   * @param failureMsg
   */
  public final void readFail(String failureMsg)
  {
    setStatus(BStatus.makeFault(getStatus(), true));
    setFaultCause(failureMsg);
  }

  /**
   * Normalize the encoded data into the pollable's data structure.
   *
   * @param encodedValue
   * @param status
   * @param cx
   */
  public final void fromEncodedValue(byte[] encodedValue, BStatus status, Context cx)
  {
    try
    {
      Property prop = lookupBacnetProperty(((PollListEntry)cx).getPropertyId());
      BInteger asnType = (BInteger)prop.getFacets().getFacet(ASN_TYPE);
      BValue v = AsnUtil.fromAsn(asnType.getInt(), encodedValue, get(prop));
      BacUtil.set(this, prop, v, noWrite);
      readOk();
    }
    catch (AsnException e)
    {
      readFail(e.toString());
      if (log.isLoggable(Level.FINE))
      {
        log.log(Level.FINE, "Exception decoding value for " + this
          + " [" + cx + "]:" + ByteArrayUtil.toHexString(encodedValue), e);
      }
    }
    catch (Exception e)
    {
      plog.log(Level.SEVERE, "Exception occurred in fromEncodedValue", e);
    }
  }

  /**
   * Get the list of poll list entries for this pollable.
   * The first entry for points must be the configured property.
   *
   * @return the list of poll list entries.
   */
  public final PollListEntry[] getPollListEntries()
  {
    return polledProperties.toArray(new PollListEntry[0]);
  }

////////////////////////////////////////////////////////////////
// Helper methods
////////////////////////////////////////////////////////////////

  private void checkConfig()
  {
    BBacnetConfigDeviceExt config = null;
    BComplex parent = getParent();
    while (parent != null)
    {
      if (parent instanceof BBacnetConfigDeviceExt)
      {
        config = (BBacnetConfigDeviceExt)parent;
        break;
      }
      parent = parent.getParent();
    }
    this.config = config;
  }

  @SuppressWarnings("unchecked")
  private void readArray(BBacnetDevice device, int propertyId, BBacnetArray array)
    throws BacnetException
  {
    BBacnetAddress address = device.getAddress();
    BBacnetObjectIdentifier objectId = getObjectId();

    // Try to read the entire array in a single readProperty.
    try
    {
      byte[] encodedValue = client().readProperty(address, objectId, propertyId);
      AsnUtil.fromAsn(BacnetConst.ASN_BACNET_ARRAY, encodedValue, array);
      return;
    }
    catch (Exception e)
    {
      if (log.isLoggable(Level.FINE))
      {
        log.log(
          Level.FINE,
          "Exception reading array property " + BBacnetPropertyIdentifier.tag(propertyId)
            + " in object " + this
            + ": " + e
            + "\n building array in groups...",
          e);
      }
    }

    // Read the array size.
    int arraySize = AsnUtil.fromAsnUnsignedInt(client().readProperty(address, objectId, propertyId, 0));

    boolean readOk = false;
    ByteArrayOutputStream os = new ByteArrayOutputStream();
    int index = 1;
    if (device.isServiceSupported("readPropertyMultiple"))
    {
      // Attempt to read chunks of the array using readPropertyMultiple
      try
      {
        BTypeSpec arrTypeSpec = array.getArrayTypeSpec();
        int elemSize = AsnUtil.getSize(arrTypeSpec);
        int hdrSize = 9; // CxACK (4) + objId (5)
        int elemHdr = 8; // propId (3) + index (3) + open/close tags (2) +
        int maxAPDUSize = device.getMaxAPDULengthAccepted();
        int myMax = BBacnetNetwork.localDevice().getMaxAPDULengthAccepted();
        if (maxAPDUSize > myMax)
        {
          maxAPDUSize = myMax;
        }
        int safetyFactor = 10;
        int elemsPerRead = (maxAPDUSize - hdrSize - safetyFactor) / (elemSize + elemHdr);

        do
        {
          @SuppressWarnings("rawtypes") Vector refs = new Vector();
          for (int i = index; i < (index + elemsPerRead) && i <= arraySize; i++)
          {
            refs.add(new NBacnetPropertyReference(propertyId, i));
          }
          @SuppressWarnings("rawtypes") Vector results = client().readPropertyMultiple(address, objectId, refs);
          for (Object result : results)
          {
            NReadPropertyResult rpr = (NReadPropertyResult)result;
            byte[] val = rpr.getPropertyValue();
            os.write(val, 0, val.length);
            index++;
          }
        } while (index <= arraySize);
        readOk = true;
      }
      catch (Exception e)
      {
        if (log.isLoggable(Level.FINE))
        {
          log.log(
            Level.FINE,
            "Exception reading array property " + BBacnetPropertyIdentifier.tag(propertyId)
              + " in object " + this + " in groups"
              + ": " + e
              + "\n building array element by element...",
            e);
        }
      }
    }

    if (!readOk)
    {
      // Read the array, one element at a time.
      // FIXX: Maybe later, try to read multiple on several at a time,
      // to speed things up a little.
      for (int i = index; i <= arraySize; i++)
      {
        byte[] encodedElement = client().readProperty(address, objectId, propertyId, i);
        os.write(encodedElement, 0, encodedElement.length);
      }
    }

    // Now, put it together.
    AsnUtil.fromAsn(BacnetConst.ASN_BACNET_ARRAY, os.toByteArray(), array);
  }

  /**
   * Read and populate a <code>BBacnetArray</code> for inclusion in this
   * BBacnetObject.  This method should not throw any exception.
   *
   * @param device
   * @param a
   * @param propId
   * @param pi
   * @return true if the array was fully read, false if there was a problem.
   */
  private boolean readArray(BBacnetDevice device, BBacnetArray a, int propId, PropertyInfo pi)
  {
    try
    {
      BBacnetAddress address = device.getAddress();
      BBacnetObjectIdentifier objectId = getObjectId();
      int asize = 0;
      try
      {
        asize = AsnUtil.fromAsnUnsignedInt(client().readProperty(address, objectId, propId, 0));
      }
      catch (Exception e)
      {
        if (log.isLoggable(Level.FINE))
        {
          log.log(Level.FINE, "Cannot get array size", e);
        }
        return false;
      }
      ByteArrayOutputStream os = new ByteArrayOutputStream();
      int i = 1;
      try
      {
        for (i = 1; i <= asize; i++)
        {
          byte[] encodedElement = client().readProperty(address,
            objectId,
            propId,
            i);
          os.write(encodedElement, 0, encodedElement.length);
        }
        byte[] encodedArray = os.toByteArray();
        AsnInputStream in = new AsnInputStream(encodedArray);
        a.readAsn(in);
        return true;
      }
      catch (Exception e)
      {
        if (log.isLoggable(Level.FINE))
        {
          log.log(Level.FINE, "Exception reading array element " + i + ":" + e, e);
        }
        return false;
      }
    }
    catch (Throwable t)
    {
      if (log.isLoggable(Level.FINE))
      {
        log.log(Level.FINE, "Unable to build BacnetArray for property " + propId, t);
      }
      return false;
    }
  }

  protected void updatePriorityArrayType(Property priorityArrayProperty, Type newType)
  {
    if (priorityArrayProperty == null)
    {
      return;
    }

    BValue priorityArray = get(priorityArrayProperty);
    if (!(priorityArray instanceof BBacnetArray) ||
        ((BBacnetArray) priorityArray).getArrayTypeSpec().equals(BBacnetPriorityValue.TYPE.getTypeSpec()))
    {
      // Replace with an array of the appropriate optional type
      set(priorityArrayProperty, new BBacnetArray(newType, 16), noWrite);
    }
  }

////////////////////////////////////////////////////////////////
// Poll Support
////////////////////////////////////////////////////////////////

  protected void buildPolledProperties()
  {
    BBacnetPoll pollService = (BBacnetPoll)network().getPollService(this);
    if (isSubscribed()) pollService.unsubscribe(this);
    SlotCursor<Property> sc = getProperties();
    BInteger pId = null;
    if (polledProperties.size() > 0)
      polledProperties.clear();

    while (sc.next())
    {
      Property p = sc.property();
      pId = (BInteger)p.getFacets().getFacet(PID);
      if ((pId != null) && shouldPoll(pId.getInt()))
        polledProperties.add(new PollListEntry(getObjectId(), pId.getInt(), device(), this));
    }
    if (isSubscribed()) pollService.subscribe(this);
  }


////////////////////////////////////////////////////////////////
//  BILoadable support
////////////////////////////////////////////////////////////////
  
  private void updateProperties(BBacnetDevice device, @SuppressWarnings("rawtypes") Iterator it)
  {
    synchronized (UPLOAD_LOCK)
    {
      while (it.hasNext())
      {
        NReadPropertyResult rpr = (NReadPropertyResult)it.next();
        int propId = rpr.getPropertyId();
        Property prop = lookupBacnetProperty(propId);   // see if we already have it
        try
        {
          if (prop == null)
          {
            // We need to create a new Property for this one.
            if (!rpr.isError())
            {
              // Get the property meta-data.
              PropertyInfo propInfo = getPropertyInfo(device, propId);

              // Decode the property value.
              BValue value = AsnUtil.asnToValue(propInfo, rpr.getPropertyValue());

              // Add the property.
              String name = SlotPath.escape(propInfo.getName());
              prop = add(name, value, 0, makeFacets(propInfo, value), null);
              if (shouldPoll(propId))
                polledProperties.add(new PollListEntry(getObjectId(), propId, device(), this));

              // For unknown propertyIds, check to see if we need to add this propertyId
              // to our device's enumeration list.
              if (!device.getEnumerationList().getPropertyIdRange().isOrdinal(propId))
              {
                device.getEnumerationList().addNewPropertyId(propInfo.getName(), propId);
              }
            } // rpr !error
          } // prop == null
          else
          {
            // We already have this Property, so set it from the encoded value.
            if (rpr.isError())
            {
              if (log.isLoggable(Level.FINE))
              {
                log.fine("Error uploading property " + prop + ":" + rpr.getPropertyAccessError());
              }
            }
            else
            {
              set(prop,
                AsnUtil.fromAsn(((BInteger)prop.getFacets().getFacet(ASN_TYPE)).getInt(),
                  rpr.getPropertyValue(),
                  get(prop)),
                noWrite);
            }
          } // else (prop != null)
        } // try
        catch (AsnException e)
        {
          log.info("Unable to convert encoded value: prop=" + prop + ", id=" + propId
            + ", val=" + ByteArrayUtil.toHexString(rpr.getPropertyValue()) + "\n" + e);
        }
        catch (Exception e)
        {          
          log.info("Unable to add/update property: prop=" + prop + ", id=" + propId
            + ", val=" + ByteArrayUtil.toHexString(rpr.getPropertyValue()) + "\n" + e);
          if (log.isLoggable(Level.FINE))
          {
            log.log(Level.FINE, "Stack Trace: ", e);
          }
        }
      } // while
    }
    if (log.isLoggable(Level.FINEST))
    {
      log.finest(device.getName() + " object updateProperties execution finish.");
    }
  }

  /**
   * Upload all properties individually.
   */
  private void uploadIndividual(BBacnetDevice device, NReadAccessSpec spec)
  {
    PropertyReference[] refs = spec.getListOfPropertyReferences();
    for (int i = 0; i < refs.length; i++)
    {
      int propId = refs[i].getPropertyId();
      try
      {
        Property prop = lookupBacnetProperty(propId);
        if (prop != null)
          readProperty(prop);
        else
        {
          byte[] encodedValue = null;
          PropertyInfo propInfo = getPropertyInfo(device, propId);
          String name = SlotPath.escape(propInfo.getName());
          try
          {
            encodedValue = client().readProperty(device.getAddress(),
              getObjectId(),
              propId);
            BValue value = AsnUtil.asnToValue(propInfo, encodedValue);
            prop = add(name, value, 0, makeFacets(propInfo, value), noWrite);
          }
          catch (BacnetException e)
          {
            if (e instanceof ErrorException)
            {
              if (((ErrorException)e).getErrorType().getErrorCode() == BBacnetErrorCode.UNKNOWN_PROPERTY)
              {
                if (log.isLoggable(Level.FINE))
                {
                  log.fine("Unknown Property " + propId + " in object " + getObjectId() + ": " + e);
                }
                continue;
              }
            }

            // Try to read the array bit by bit.
            if (propInfo.isArray())
            {
              BBacnetArray a = new BBacnetArray();
              a.setArrayTypeSpec(BTypeSpec.make(propInfo.getType()));
              readArray(device, a, propId, propInfo);
              prop = add(name, a, 0, makeFacets(propInfo, a), noWrite);
            }

            log.info("BacnetException uploading propertyId " + propId + " in object " + getObjectId() + ": " + e);
          }

          // add to poll list
          if (shouldPoll(propId))
            polledProperties.add(new PollListEntry(getObjectId(), propId, device(), this));

        } // else (a new property)

      } // outer try
      catch (TransactionException e)
      {
        if (log.isLoggable(Level.FINE))
        {
          log.fine("TransactionException uploading object " + getObjectId() + " in " + device() + ": " + e);
        }
        break;
      }
      catch (Exception e)
      {
        if (log.isLoggable(Level.FINE))
        {
          log.fine("Exception uploading propertyId " + propId + " in object " + getObjectId() + ": " + e);
        }
      }
    }
  }

  /**
   * Get a PropertyInfo object containing metadata about this property.
   *
   * @param device
   * @param propId the property ID.
   * @return a PropertyInfo.
   */
  private PropertyInfo getPropertyInfo(BBacnetDevice device, int propId)
  {
    // First, try the list of Bacnet-defined properties.
    PropertyInfo propInfo = device.getPropertyInfo(getObjectId().getObjectType(), propId);

    // If still nothing, just create an "unknown proprietary" PropertyInfo.
    if (propInfo == null)
    {
      propInfo = new PropertyInfo(BBacnetPropertyIdentifier.tag(propId), propId, AsnConst.ASN_UNKNOWN_PROPRIETARY);
    }

    // Return what we have.
    return propInfo;
  }


////////////////////////////////////////////////////////////////
// Spy
////////////////////////////////////////////////////////////////

  /**
   * Spy.
   */
  public void spy(SpyWriter out)
    throws Exception
  {
    super.spy(out);
    out.startProps();
    out.trTitle("BacnetObject", 2);
    out.prop("config", config);
    if (polledProperties != null)
    {
      int siz = polledProperties.size();
      out.prop("polledProperties", siz);
      for (int i = 0; i < siz; i++)
        out.prop("polledProperties[" + i + "]:", polledProperties.get(i).debugString());
    }
    else
      out.prop("polledProperties", "NULL");
    out.prop("propDataMap", propDataMap.size());
    Iterator<Map.Entry<BFacets,BacnetPropertyData>> it = propDataMap.entrySet().iterator();
    while (it.hasNext())
    {
      Map.Entry<BFacets,BacnetPropertyData> entry = it.next();
      out.prop(entry.getKey(), entry.getValue());
    }
    out.endProps();
  }


//////////////////////////////////////////////////////////////////
//   Bacnet Property management
//////////////////////////////////////////////////////////////////

  protected Property lookupBacnetProperty(int propId)
  {
    SlotCursor<Property> c = getProperties();
    while (c.next())
    {
      try
      {
        Property property = c.property();
        BInteger propertyIdFacet = (BInteger)property.getFacets().getFacet(PID);
        if (propertyIdFacet != null &&
            propertyIdFacet.getInt() == propId)
        {
          return property;
        }
      }
      catch (Exception ignored)
      {
        //Keep looking
      }
    }

    return null;
  }


////////////////////////////////////////////////////////////////
// Presentation
////////////////////////////////////////////////////////////////

  public BIcon getIcon()
  {
    return icon;
  }

  private static final BIcon icon = BIcon.make("module://bacnet/com/tridium/bacnet/ui/icons/bacObject.png");

////////////////////////////////////////////////////////////////
// Constants
////////////////////////////////////////////////////////////////

  protected static final Lexicon lex = Lexicon.make("bacnet");
  public static final Logger log = Logger.getLogger("bacnet.client");
  public static final Logger plog = Logger.getLogger("bacnet.point");

  private static final Map<Integer, Array<TypeInfo>> byObjectType = new HashMap<>();
  private static boolean initialized = false;

  private static final BRelTime WRITE_PROPS_DELAY = BRelTime.make(20); // 20ms delay
  private static final Object UPLOAD_LOCK = new Object();

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  /**
   * List of BACnetPropertyReferences for the properties of this object.
   */
  protected volatile ArrayList<PollListEntry> polledProperties = new ArrayList<>();

  private BBacnetConfigDeviceExt config;
  private final Map<BFacets, BacnetPropertyData> propDataMap = new HashMap<>();

  private final Map<PropertyWrite, PropertyWrite> writeProps = new HashMap<>();
  private Clock.Ticket writePropsTicket = null;

////////////////////////////////////////////////////////////////
// Initialization
////////////////////////////////////////////////////////////////

  static void init()
  {
    TypeInfo base = BBacnetObject.TYPE.getTypeInfo();
    TypeInfo[] types = Sys.getRegistry().getConcreteTypes(base);
    for (int i = 0; i < types.length; i++)
    {
      if (types[i].equals(base)) continue;
      BBacnetObject o = (BBacnetObject)types[i].getInstance();
      int objTypOrd = o.getObjectType().getOrdinal();
      Array<TypeInfo> cur = byObjectType.get(objTypOrd);
      if (cur == null)
        cur = new Array<>(TypeInfo.class);
      cur.add(types[i]);
      byObjectType.put(objTypOrd, cur);
    }
    initialized = true;
  }

////////////////////////////////////////////////////////////////
// Facets support
////////////////////////////////////////////////////////////////

  /**
   * Property ID facet name.
   */
  public static final String PID = "pId";
  /**
   * Asn Type facet name.
   */
  public static final String ASN_TYPE = "asn";
  /**
   * Non-Bacnet property metadata.
   */
  private static final BacnetPropertyData NOT_BACNET_PROPERTY = new BacnetPropertyData(NOT_USED, 0);

  /**
   * Make a BFacets with property ID and Asn type.
   */
  protected static BFacets makeFacets(int propertyId, int asnType)
  {
    HashMap<String, BIDataValue> map = new HashMap<>();
    map.put(PID, BInteger.make(propertyId));
    map.put(ASN_TYPE, BInteger.make(asnType));
    return BFacets.make(map);
  }

  /**
   * Make a BFacets with property ID, Asn type, and a Map
   * which contains additional info.
   * Used for bit strings, to name the bits.
   */
  protected static BFacets makeFacets(int propertyId, int asnType, Map<String, BIDataValue> m)
  {
    HashMap<String, BIDataValue> map = new HashMap<>(m);
    map.put(PID, BInteger.make(propertyId));
    map.put(ASN_TYPE, BInteger.make(asnType));
    return BFacets.make(map);
  }

  /**
   * Make a BFacets with property ID, Asn type, and two arrays of additional
   * keys and values.
   */
  protected static BFacets makeFacets(int propertyId, int asnType, String[] keys, BIDataValue[] values)
  {
    if (keys.length != values.length)
      throw new IllegalArgumentException();

    String[] k = new String[keys.length + 2];
    System.arraycopy(keys, 0, k, 0, keys.length);
    k[keys.length] = PID;
    k[keys.length + 1] = ASN_TYPE;

    BIDataValue[] v = new BIDataValue[values.length + 2];
    System.arraycopy(values, 0, v, 0, values.length);
    v[values.length] = BInteger.make(propertyId);
    v[values.length + 1] = BInteger.make(asnType);

    return BFacets.make(k, v);
  }

  /**
   * Make a BFacets from the given <code>PropertyInfo</code>.
   * Used in dynamic creation of properties.
   */
  protected static BFacets makeFacets(PropertyInfo info, BValue value)
  {
    HashMap<String, BIDataValue> map;
    if (info.isBitString())
      map = new HashMap<>(BacnetBitStringUtil.getBitStringMap(info.getBitStringName()));
    else
      map = new HashMap<>();
    map.put(PID, BInteger.make(info.getId()));
    map.put(ASN_TYPE, BInteger.make(info.getAsnType()));
    return BFacets.make(map);
  }

  protected BacnetPropertyData getPropertyData(Property prop)
  {
    BFacets f = prop.getFacets();
    if (f == null) return NOT_BACNET_PROPERTY;
    if (f.geti(PID, NOT_USED) == NOT_USED) return NOT_BACNET_PROPERTY;

    if (!prop.isDynamic())
    {
      BacnetPropertyData d = propDataMap.get(f);
      if (d == null)
      {
        d = makePropertyData(f);
        propDataMap.put(f, d);
        return d;
      }
    }
    return makePropertyData(f);
  }

  /**
   * Make a new metadata container object from the given BFacets.
   */
  private static BacnetPropertyData makePropertyData(BFacets f)
  {
    int propertyId = NOT_USED;
    int asnType = 0;
    BObject s;
    if ((s = f.getFacet(PID)) != null)
    {
      propertyId = ((BInteger)s).getInt();
    }
    if ((s = f.getFacet(ASN_TYPE)) != null)
    {
      asnType = ((BInteger)s).getInt();
    }
    return BacnetPropertyData.make(propertyId, asnType);
  }


////////////////////////////////////////////////////////////////
// BacnetPropertyData
////////////////////////////////////////////////////////////////

  public static class BacnetPropertyData
  {
    private BacnetPropertyData(int propertyId, int asnType)
    {
      this.propertyId = propertyId;
      this.asnType = asnType;
    }

    static BacnetPropertyData make(int pid, int asn)
    {
      if ((pid == NOT_USED) && (asn == 0)) return NOT_BACNET_PROPERTY;
      return new BacnetPropertyData(pid, asn);
    }

    public int getPropertyId()
    {
      return propertyId;
    }

    public int getAsnType()
    {
      return asnType;
    }

    int propertyId;
    int asnType;
  }

  private static class PropertyWrite
  {
    public final BacnetPropertyData propData;
    public final int arrayIndex;
    public byte[] propValue;
    public final Property configProp;

    private final int hash;

    public PropertyWrite(BacnetPropertyData propData, Property configProp)
    {
      this.propData = propData;
      this.arrayIndex = NOT_USED;
      propValue = null;
      this.configProp = configProp;
      hash = Objects.hash(propData.propertyId, arrayIndex);
    }

    public PropertyWrite(BacnetPropertyData propData, int arrayIndex, byte[] propValue)
    {
      this.propData = propData;
      this.arrayIndex = arrayIndex;
      this.propValue = propValue;
      configProp = null;
      hash = Objects.hash(propData.propertyId, arrayIndex);
    }

    @Override
    public boolean equals(Object o)
    {
      if (o == null || getClass() != o.getClass())
      {
        return false;
      }
      PropertyWrite that = (PropertyWrite)o;
      return propData.propertyId == that.propData.propertyId && arrayIndex == that.arrayIndex;
    }

    @Override
    public int hashCode()
    {
      return hash;
    }
  }
}
