/*
 * Copyright 2001 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.config;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import javax.baja.bacnet.BBacnetDevice;
import javax.baja.bacnet.BBacnetNetwork;
import javax.baja.bacnet.BBacnetObject;
import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetDateTime;
import javax.baja.bacnet.datatypes.BBacnetObjectIdentifier;
import javax.baja.bacnet.datatypes.BBacnetOctetString;
import javax.baja.bacnet.datatypes.BBacnetUnsigned;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetFileAccessMethod;
import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ErrorException;
import javax.baja.bacnet.io.FileData;
import javax.baja.file.BIFile;
import javax.baja.file.BLocalFileStore;
import javax.baja.naming.BOrd;
import javax.baja.naming.NullOrdException;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.Action;
import javax.baja.sys.BBlob;
import javax.baja.sys.BDynamicEnum;
import javax.baja.sys.BEnumRange;
import javax.baja.sys.BFacets;
import javax.baja.sys.BObject;
import javax.baja.sys.BStruct;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.bacnet.asn.AsnInputStream;
import com.tridium.bacnet.asn.AsnOutputStream;
import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.datatypes.BReadFileConfig;
import com.tridium.bacnet.datatypes.BWriteFileConfig;
import com.tridium.bacnet.stack.BBacnetStack;
import com.tridium.bacnet.stack.client.BBacnetClientLayer;

/**
 * @author Craig Gemmill
 * @version $Revision: 7$ $Date: 12/10/01 9:26:02 AM$
 * @creation 30 Jan 01
 * @since Niagara 3 Bacnet 1.0
 */
@NiagaraType
@NiagaraProperty(
  name = "objectId",
  type = "BBacnetObjectIdentifier",
  defaultValue = "BBacnetObjectIdentifier.make(BBacnetObjectType.FILE)",
  flags = Flags.SUMMARY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.OBJECT_IDENTIFIER, ASN_OBJECT_IDENTIFIER)"),
  override = true
)
@NiagaraProperty(
  name = "objectType",
  type = "BEnum",
  defaultValue = "BDynamicEnum.make(BBacnetObjectType.FILE, BEnumRange.make(BBacnetObjectType.TYPE))",
  flags = Flags.READONLY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.OBJECT_TYPE, ASN_ENUMERATED)"),
  override = true
)
@NiagaraProperty(
  name = "fileType",
  type = "String",
  defaultValue = "",
  flags = Flags.READONLY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.FILE_TYPE, ASN_CHARACTER_STRING)")
)
@NiagaraProperty(
  name = "fileSize",
  type = "BBacnetUnsigned",
  defaultValue = "BBacnetUnsigned.make(0)",
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.FILE_SIZE, ASN_UNSIGNED)")
)
@NiagaraProperty(
  name = "modificationDate",
  type = "BBacnetDateTime",
  defaultValue = "new BBacnetDateTime()",
  flags = Flags.READONLY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.MODIFICATION_DATE, ASN_CONSTRUCTED_DATA)")
)
/*
 has this file been archived?
 TRUE if no changes have been made since the last time
 the object was archived.
 */
@NiagaraProperty(
  name = "archive",
  type = "boolean",
  defaultValue = "false",
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.ARCHIVE, ASN_BOOLEAN)")
)
@NiagaraProperty(
  name = "readOnly",
  type = "boolean",
  defaultValue = "true",
  flags = Flags.READONLY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.READ_ONLY, ASN_BOOLEAN)")
)
@NiagaraProperty(
  name = "fileAccessMethod",
  type = "BBacnetFileAccessMethod",
  defaultValue = "BBacnetFileAccessMethod.streamAccess",
  flags = Flags.READONLY,
  facets = @Facet("makeFacets(BBacnetPropertyIdentifier.FILE_ACCESS_METHOD, ASN_ENUMERATED)")
)
/*
 the ord to the local storage for this file's contents.
 */
@NiagaraProperty(
  name = "fileOrd",
  type = "BOrd",
  defaultValue = "BOrd.NULL",
  flags = Flags.DEFAULT_ON_CLONE,
  facets = @Facet(name = "BFacets.TARGET_TYPE", value = "\"baja:IFile\"")
)
/*
 Read file data and return the data as a BBlob.
 */
@NiagaraAction(
  name = "read",
  returnType = "BBlob"
)
/*
 Write file data given as a BBlob.
 */
@NiagaraAction(
  name = "write",
  parameterType = "BBlob",
  defaultValue = "BBlob.DEFAULT"
)
/*
 Read 'count' bytes or records from the file referenced by this
 File object from the Bacnet device, beginning at the record
 or byte designated by 'start'.
 Store it locally in the file referenced by filename.
 */
@NiagaraAction(
  name = "readFile",
  parameterType = "BStruct",
  defaultValue = "new BReadFileConfig()",
  flags = Flags.HIDDEN
)
/*
 Write to the file referenced by this File object.
 Use the file given by the argument as the source file.
 */
@NiagaraAction(
  name = "writeFile",
  parameterType = "BStruct",
  defaultValue = "new BWriteFileConfig()",
  flags = Flags.HIDDEN
)
public class BBacnetFile
  extends BBacnetObject
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.config.BBacnetFile(1332449050)1.0$ @*/
/* Generated Thu Jun 02 14:30:00 EDT 2022 by Slot-o-Matic (c) Tridium, Inc. 2012-2022 */

  //region Property "objectId"

  /**
   * Slot for the {@code objectId} property.
   * @see #getObjectId
   * @see #setObjectId
   */
  @Generated
  public static final Property objectId = newProperty(Flags.SUMMARY, BBacnetObjectIdentifier.make(BBacnetObjectType.FILE), makeFacets(BBacnetPropertyIdentifier.OBJECT_IDENTIFIER, ASN_OBJECT_IDENTIFIER));

  //endregion Property "objectId"

  //region Property "objectType"

  /**
   * Slot for the {@code objectType} property.
   * @see #getObjectType
   * @see #setObjectType
   */
  @Generated
  public static final Property objectType = newProperty(Flags.READONLY, BDynamicEnum.make(BBacnetObjectType.FILE, BEnumRange.make(BBacnetObjectType.TYPE)), makeFacets(BBacnetPropertyIdentifier.OBJECT_TYPE, ASN_ENUMERATED));

  //endregion Property "objectType"

  //region Property "fileType"

  /**
   * Slot for the {@code fileType} property.
   * @see #getFileType
   * @see #setFileType
   */
  @Generated
  public static final Property fileType = newProperty(Flags.READONLY, "", makeFacets(BBacnetPropertyIdentifier.FILE_TYPE, ASN_CHARACTER_STRING));

  /**
   * Get the {@code fileType} property.
   * @see #fileType
   */
  @Generated
  public String getFileType() { return getString(fileType); }

  /**
   * Set the {@code fileType} property.
   * @see #fileType
   */
  @Generated
  public void setFileType(String v) { setString(fileType, v, null); }

  //endregion Property "fileType"

  //region Property "fileSize"

  /**
   * Slot for the {@code fileSize} property.
   * @see #getFileSize
   * @see #setFileSize
   */
  @Generated
  public static final Property fileSize = newProperty(0, BBacnetUnsigned.make(0), makeFacets(BBacnetPropertyIdentifier.FILE_SIZE, ASN_UNSIGNED));

  /**
   * Get the {@code fileSize} property.
   * @see #fileSize
   */
  @Generated
  public BBacnetUnsigned getFileSize() { return (BBacnetUnsigned)get(fileSize); }

  /**
   * Set the {@code fileSize} property.
   * @see #fileSize
   */
  @Generated
  public void setFileSize(BBacnetUnsigned v) { set(fileSize, v, null); }

  //endregion Property "fileSize"

  //region Property "modificationDate"

  /**
   * Slot for the {@code modificationDate} property.
   * @see #getModificationDate
   * @see #setModificationDate
   */
  @Generated
  public static final Property modificationDate = newProperty(Flags.READONLY, new BBacnetDateTime(), makeFacets(BBacnetPropertyIdentifier.MODIFICATION_DATE, ASN_CONSTRUCTED_DATA));

  /**
   * Get the {@code modificationDate} property.
   * @see #modificationDate
   */
  @Generated
  public BBacnetDateTime getModificationDate() { return (BBacnetDateTime)get(modificationDate); }

  /**
   * Set the {@code modificationDate} property.
   * @see #modificationDate
   */
  @Generated
  public void setModificationDate(BBacnetDateTime v) { set(modificationDate, v, null); }

  //endregion Property "modificationDate"

  //region Property "archive"

  /**
   * Slot for the {@code archive} property.
   * has this file been archived?
   * TRUE if no changes have been made since the last time
   * the object was archived.
   * @see #getArchive
   * @see #setArchive
   */
  @Generated
  public static final Property archive = newProperty(0, false, makeFacets(BBacnetPropertyIdentifier.ARCHIVE, ASN_BOOLEAN));

  /**
   * Get the {@code archive} property.
   * has this file been archived?
   * TRUE if no changes have been made since the last time
   * the object was archived.
   * @see #archive
   */
  @Generated
  public boolean getArchive() { return getBoolean(archive); }

  /**
   * Set the {@code archive} property.
   * has this file been archived?
   * TRUE if no changes have been made since the last time
   * the object was archived.
   * @see #archive
   */
  @Generated
  public void setArchive(boolean v) { setBoolean(archive, v, null); }

  //endregion Property "archive"

  //region Property "readOnly"

  /**
   * Slot for the {@code readOnly} property.
   * @see #getReadOnly
   * @see #setReadOnly
   */
  @Generated
  public static final Property readOnly = newProperty(Flags.READONLY, true, makeFacets(BBacnetPropertyIdentifier.READ_ONLY, ASN_BOOLEAN));

  /**
   * Get the {@code readOnly} property.
   * @see #readOnly
   */
  @Generated
  public boolean getReadOnly() { return getBoolean(readOnly); }

  /**
   * Set the {@code readOnly} property.
   * @see #readOnly
   */
  @Generated
  public void setReadOnly(boolean v) { setBoolean(readOnly, v, null); }

  //endregion Property "readOnly"

  //region Property "fileAccessMethod"

  /**
   * Slot for the {@code fileAccessMethod} property.
   * @see #getFileAccessMethod
   * @see #setFileAccessMethod
   */
  @Generated
  public static final Property fileAccessMethod = newProperty(Flags.READONLY, BBacnetFileAccessMethod.streamAccess, makeFacets(BBacnetPropertyIdentifier.FILE_ACCESS_METHOD, ASN_ENUMERATED));

  /**
   * Get the {@code fileAccessMethod} property.
   * @see #fileAccessMethod
   */
  @Generated
  public BBacnetFileAccessMethod getFileAccessMethod() { return (BBacnetFileAccessMethod)get(fileAccessMethod); }

  /**
   * Set the {@code fileAccessMethod} property.
   * @see #fileAccessMethod
   */
  @Generated
  public void setFileAccessMethod(BBacnetFileAccessMethod v) { set(fileAccessMethod, v, null); }

  //endregion Property "fileAccessMethod"

  //region Property "fileOrd"

  /**
   * Slot for the {@code fileOrd} property.
   * the ord to the local storage for this file's contents.
   * @see #getFileOrd
   * @see #setFileOrd
   */
  @Generated
  public static final Property fileOrd = newProperty(Flags.DEFAULT_ON_CLONE, BOrd.NULL, BFacets.make(BFacets.TARGET_TYPE, "baja:IFile"));

  /**
   * Get the {@code fileOrd} property.
   * the ord to the local storage for this file's contents.
   * @see #fileOrd
   */
  @Generated
  public BOrd getFileOrd() { return (BOrd)get(fileOrd); }

  /**
   * Set the {@code fileOrd} property.
   * the ord to the local storage for this file's contents.
   * @see #fileOrd
   */
  @Generated
  public void setFileOrd(BOrd v) { set(fileOrd, v, null); }

  //endregion Property "fileOrd"

  //region Action "read"

  /**
   * Slot for the {@code read} action.
   * Read file data and return the data as a BBlob.
   * @see #read()
   */
  @Generated
  public static final Action read = newAction(0, null);

  /**
   * Invoke the {@code read} action.
   * Read file data and return the data as a BBlob.
   * @see #read
   */
  @Generated
  public BBlob read() { return (BBlob)invoke(read, null, null); }

  //endregion Action "read"

  //region Action "write"

  /**
   * Slot for the {@code write} action.
   * Write file data given as a BBlob.
   * @see #write(BBlob parameter)
   */
  @Generated
  public static final Action write = newAction(0, BBlob.DEFAULT, null);

  /**
   * Invoke the {@code write} action.
   * Write file data given as a BBlob.
   * @see #write
   */
  @Generated
  public void write(BBlob parameter) { invoke(write, parameter, null); }

  //endregion Action "write"

  //region Action "readFile"

  /**
   * Slot for the {@code readFile} action.
   * Read 'count' bytes or records from the file referenced by this
   * File object from the Bacnet device, beginning at the record
   * or byte designated by 'start'.
   * Store it locally in the file referenced by filename.
   * @see #readFile(BStruct parameter)
   */
  @Generated
  public static final Action readFile = newAction(Flags.HIDDEN, new BReadFileConfig(), null);

  /**
   * Invoke the {@code readFile} action.
   * Read 'count' bytes or records from the file referenced by this
   * File object from the Bacnet device, beginning at the record
   * or byte designated by 'start'.
   * Store it locally in the file referenced by filename.
   * @see #readFile
   */
  @Generated
  public void readFile(BStruct parameter) { invoke(readFile, parameter, null); }

  //endregion Action "readFile"

  //region Action "writeFile"

  /**
   * Slot for the {@code writeFile} action.
   * Write to the file referenced by this File object.
   * Use the file given by the argument as the source file.
   * @see #writeFile(BStruct parameter)
   */
  @Generated
  public static final Action writeFile = newAction(Flags.HIDDEN, new BWriteFileConfig(), null);

  /**
   * Invoke the {@code writeFile} action.
   * Write to the file referenced by this File object.
   * Use the file given by the argument as the source file.
   * @see #writeFile
   */
  @Generated
  public void writeFile(BStruct parameter) { invoke(writeFile, parameter, null); }

  //endregion Action "writeFile"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetFile.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  public BBacnetFile()
  {
  }


////////////////////////////////////////////////////////////////
//  BComponent
////////////////////////////////////////////////////////////////

  /**
   * Register with the Bacnet service when this component is started.
   */
  public void started()
    throws Exception
  {
    super.started();
    getFile();
  }

  /**
   * Stopped.
   */
  public void stopped()
  {
    file = null;
  }

  /**
   * Property Changed.
   */
  public void changed(Property p, Context cx)
  {
    super.changed(p, cx);
    if (!isRunning()) return;
    if (p.equals(fileOrd))
    {
      getFile();
    }
  }


////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  public String toString(Context context)
  {
    return getObjectId().toString(context) + " local: " + getFileOrd();
  }


////////////////////////////////////////////////////////////////
// Actions
////////////////////////////////////////////////////////////////

  public static byte[] readFile(BBacnetDevice device,
                                BBacnetObjectIdentifier objectId)
    throws BacnetException
  {
    int fileAccessMethod = AsnUtil.fromAsnEnumerated(
      client().readProperty(
        device.getAddress(),
        objectId,
        BBacnetPropertyIdentifier.FILE_ACCESS_METHOD));

    if (fileAccessMethod == BBacnetFileAccessMethod.STREAM_ACCESS)
    {
      int fileSize = -1;
      int requestedOctetCount = Integer.MAX_VALUE;
      try
      {
        // Stream access: just read the file & return it.
        fileSize = AsnUtil.fromAsnUnsignedInt(
          client().readProperty(
            device.getAddress(),
            objectId,
            BBacnetPropertyIdentifier.FILE_SIZE));
        requestedOctetCount = fileSize;
      }
      catch (ErrorException e)
      {
        if (e.getErrorType().getErrorCode() != BBacnetErrorCode.UNKNOWN_FILE_SIZE)
        {
          throw e;
        }
      }

      return readFileDataStream(device, objectId, fileSize, 0, requestedOctetCount);
    }
    else
    {
      // Record access: read the file records, then put them together into
      // one big byte array.  The byte array will have to be ASN-decoded to
      // distinguish the individual records.
      int recordCount = AsnUtil.fromAsnUnsignedInt(
        client().readProperty(
          device.getAddress(),
          objectId,
          BBacnetPropertyIdentifier.RECORD_COUNT));
      return readFileDataRecord(device, objectId, 0, recordCount);
    }
  }

  @SuppressWarnings("unused")
  public BBlob doRead()
  {
    // Read file data.
    BBacnetDevice device = device();
    if (device == null)
    {
      throw new BajaRuntimeException(this + ": device not found");
    }

    try
    {
      byte[] fileData = readFile(device, getObjectId());
      return BBlob.make(fileData);
    }
    catch (BacnetException e)
    {
      log.log(Level.SEVERE, "Unable to read file contents for " + getObjectId() + " : " + e, e);
      throw new BajaRuntimeException(e);
    }
  }

  @SuppressWarnings("unused")
  public void doReadFile(BStruct arg)
  {
    BBacnetNetwork.bacnet().postAsync(new ReadFileReq((BReadFileConfig)arg, this));
  }

  @SuppressWarnings("unused")
  public void doWriteFile(BStruct arg)
  {
    BBacnetNetwork.bacnet().postAsync(new WriteFileReq((BWriteFileConfig)arg, this));
  }

  public static void writeFile(BBacnetDevice device, BBacnetObjectIdentifier objectId, byte[] fileData)
    throws BacnetException
  {
    writeFileDataStream(device, objectId, 0, fileData);
  }

  public static void writeFile(BBacnetDevice device, BBacnetObjectIdentifier objectId, int count, BBacnetOctetString[] fileRecordData)
    throws BacnetException
  {
    writeFileDataRecord(device, objectId, 0, count, fileRecordData);
  }

  @SuppressWarnings("unused")
  public void doWrite(BBlob arg)
  {
    BBacnetDevice device = device();
    if (device == null)
    {
      throw new BajaRuntimeException(this + ": device not found");
    }

    byte[] fileData = arg.copyBytes();
    if (getFileAccessMethod() == BBacnetFileAccessMethod.streamAccess)
    {
      try
      {
        writeFileDataStream(device, getObjectId(), 0, fileData);
      }
      catch (BacnetException e)
      {
        log.log(Level.SEVERE, "Unable to write file contents for " + getObjectId() + " : " + e, e);
        throw new BajaRuntimeException(e);
      }
    }
    else
    {
      try
      {
        BBacnetOctetString[] fileRecordData = getFileRecordData(fileData);
        writeFileDataRecord(device, getObjectId(), 0, fileRecordData.length, fileRecordData);
      }
      catch (AsnException e)
      {
        log.severe("File data is not in array of encoded BACnetOctetStrings");
        throw new BajaRuntimeException(e);
      }
      catch (BacnetException e)
      {
        log.log(Level.SEVERE, "Unable to write file record contents for " + getObjectId() + ": " + e, e);
        throw new BajaRuntimeException(e);
      }
    }
  }


////////////////////////////////////////////////////////////////
// Support
////////////////////////////////////////////////////////////////

  private BIFile getFile()
  {
    try
    {
      if (!fileOrd.isEquivalentToDefaultValue(getFileOrd()))
      {
        BObject o = getFileOrd().get(this);
        if (o instanceof BIFile)
          file = (BIFile)o;
        else
          file = null;
      }
    }
    catch (Exception e)
    {
      log.log(Level.WARNING, "Unable to resolve file ord for " + this + ": " + getFileOrd(), e);
      file = null;
    }
    return file;
  }

  private static BBacnetClientLayer client()
  {
    return ((BBacnetStack)BBacnetNetwork.bacnet().getBacnetComm()).getClient();
  }

  private static byte[] readFileDataRecord(
    BBacnetDevice device,
    BBacnetObjectIdentifier objectId,
    int fileStartRecord,
    int requestedRecordCount)
      throws BacnetException
  {
    if (!device.isServiceSupported("atomicReadFile"))
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.atomicReadFile"));

    AsnOutputStream out = new AsnOutputStream();

    // If we either know we need to get it in multiple requests, or the single
    // request failed, try the multiple request approach.
    FileData ack;
    for (int i = 0; i < requestedRecordCount; i++)
    {
      ack = client().atomicReadFile(device.getAddress(),
        objectId,
        FileData.RECORD_ACCESS,
        fileStartRecord + i,
        1);
      out.writeOctetString(ack.getFileRecordData()[0]);
      if (ack.isEndOfFile()) break;
    }

    return out.toByteArray();
  }

  private static byte[] readFileDataStream(BBacnetDevice device,
                                           BBacnetObjectIdentifier objectId,
                                           int fileSize,
                                           int fileStartPosition,
                                           int requestedOctetCount)
    throws BacnetException
  {
    if (!device.isServiceSupported("atomicReadFile"))
    {
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.atomicReadFile"));
    }

    ByteArrayOutputStream data;
    if (fileSize < 0)
    {
      // File size is unknown so append until the end is indicated in the response/ack.
      data = new ByteArrayOutputStream();
    }
    else
    {
      // TODO If the fileStartPosition is greater than zero, could this byte array output stream be
      //  smaller than the full fileSize?
      data = new ByteArrayOutputStream(fileSize);

      // If the file size is known, we cannot request more than that.
      if (fileSize < requestedOctetCount)
      {
        requestedOctetCount = fileSize;
      }
    }

    int lastByte = Integer.MAX_VALUE;
    if (requestedOctetCount < Integer.MAX_VALUE)
    {
      try
      {
        lastByte = Math.addExact(fileStartPosition, requestedOctetCount);
      }
      catch (ArithmeticException ignore)
      {
      }
    }

    // Determine device's data return capability.  First take minimum of device's and our own max
    // APDU.
    int maxReturnableFileSize = device.getMaxAPDULengthAccepted();
    int myMax = BBacnetNetwork.localDevice().getMaxAPDULengthAccepted();
    if (myMax < maxReturnableFileSize)
    {
      maxReturnableFileSize = myMax;
    }

    // Then, subtract the non-data request and apdu header size.  Subtract an extra safety factor
    // because some devices abort if you get close to (but not over) the limit.
    maxReturnableFileSize -= STREAM_ACCESS_HEADER_SIZE - FILE_DATA_SAFETY_FACTOR;

    // Subtract the octet string application tag length
    maxReturnableFileSize -= getTagSize(maxReturnableFileSize);

    int start = fileStartPosition;
    int len = maxReturnableFileSize;
    FileData ack;
    do
    {
      ack = client().atomicReadFile(
        device.getAddress(),
        objectId,
        FileData.STREAM_ACCESS,
        start,
        len);
      byte[] fileData = ack.getFileData();
      data.write(fileData, /* offset */ 0, fileData.length);
      start += len;
    } while (!ack.isEndOfFile() && (start < lastByte));

    // Return the data.
    return data.toByteArray();
  }

  public static BBacnetOctetString[] getFileRecordData(byte[] fileData)
    throws AsnException
  {
    // When we read a record-oriented file, the BACnetOctetStrings are ASN encoded to a single
    // byte array where each BACnetOctetString is delimited with application tags.  Here, we split
    // that byte array back up into the BACnetOctetString file records.  This works for the restore
    // job because we only need to write the file down exactly as we read it up during the backup
    // job.  We cannot write an arbitrary file because we don't know how to split it into records.
    List<BBacnetOctetString> records = new ArrayList<>();
    AsnInputStream in = new AsnInputStream(fileData);
    while (in.available() > 0)
    {
      records.add(in.readBacnetOctetString());
    }
    return records.toArray(new BBacnetOctetString[0]);
  }

  private static void writeFileDataRecord(BBacnetDevice device,
                                          BBacnetObjectIdentifier objectId,
                                          int fileStartRecord,
                                          int recordCount,
                                          BBacnetOctetString[] fileRecordData)
    throws BacnetException
  {
    if (!device.isServiceSupported("atomicWriteFile"))
    {
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.atomicWriteFile"));
    }

    // Sanity check
    if (fileRecordData == null)
    {
      throw new IllegalArgumentException("fileRecordData is null!");
    }

    // When there are no file records to be written, just send an empty request.
    if (fileRecordData.length == 0)
    {
      client().atomicWriteFileRecord(
        device.getAddress(),
        objectId,
        fileStartRecord,
        /* count */ 0,
        fileRecordData);
      return;
    }

    // Fix count to be written if it goes beyond the last record.
    int writeCount = recordCount;
    if (fileStartRecord + recordCount > fileRecordData.length)
    {
      writeCount = fileRecordData.length - fileStartRecord;
    }

    // Determine device's data accept capability.  First take minimum of device's and our own max
    // APDU.
    int maxApdu = device.getMaxAPDULengthAccepted();
    int myMax = BBacnetNetwork.localDevice().getMaxAPDULengthAccepted();
    if (myMax < maxApdu) maxApdu = myMax;

    // Then, subtract the non-data request and apdu header size.  Subtract an extra safety factor
    // because some devices abort if you get close to (but not over) the limit.
    int maxFileDataSize = maxApdu - RECORD_ACCESS_HEADER_SIZE - FILE_DATA_SAFETY_FACTOR;

    // If we either know we need to send it in multiple requests, or the single
    // request failed, try the multiple request approach.
    int recordIndex = fileStartRecord;
    List<BBacnetOctetString> requestRecords = new ArrayList<>(writeCount);
    while ((recordIndex - fileStartRecord) < writeCount)
    {
      int start = recordIndex;
      requestRecords.clear();

      int recordSize = fileRecordData[recordIndex].length();
      int fileDataSize = getTagSize(recordSize) + recordSize;
      while (true)
      {
        requestRecords.add(fileRecordData[recordIndex]);

        recordIndex++;
        if ((recordIndex - fileStartRecord) >= writeCount)
        {
          // All records written
          break;
        }

        // Calculate the encoded length if the next record is included
        recordSize = fileRecordData[recordIndex].length();
        fileDataSize += (getTagSize(recordSize) + recordSize);
        if (fileDataSize > maxFileDataSize)
        {
          // The next record will not fit.
          break;
        }
      }

      BBacnetOctetString[] requestRecordData = requestRecords.toArray(new BBacnetOctetString[0]);
      client().atomicWriteFileRecord(
        device.getAddress(),
        objectId,
        start,
        requestRecordData.length,
        requestRecordData);
    }
  }

  private static void writeFileDataStream(BBacnetDevice device,
                                          BBacnetObjectIdentifier objectId,
                                          int fileStartPosition,
                                          byte[] fileData)
    throws BacnetException
  {
    if (!device.isServiceSupported("atomicWriteFile"))
      throw new UnsupportedOperationException(lex.getText("serviceNotSupported.atomicWriteFile"));

    int writeLength = fileData.length;

    // Determine device's data return capability.  First take minimum of device's and our own max
    // APDU.
    int maxApdu = device.getMaxAPDULengthAccepted();
    int myMax = BBacnetNetwork.localDevice().getMaxAPDULengthAccepted();
    if (myMax < maxApdu) maxApdu = myMax;

    // Then, subtract the non-data request and apdu header size.  Subtract an extra safety factor
    // because some devices abort if you get close to (but not over) the limit.
    int maxFileDataSize = maxApdu - STREAM_ACCESS_HEADER_SIZE - FILE_DATA_SAFETY_FACTOR;

    // Subtract the octet string application tag length
    maxFileDataSize -= getTagSize(maxFileDataSize);

    // If we either know we need to send it in multiple requests, or the single
    // request failed, try the multiple request approach.
    int start = fileStartPosition;
    do
    {
      int copylen = maxFileDataSize;
      if (start + copylen > writeLength) copylen = writeLength - start;
      byte[] reqFileData = new byte[copylen];
      System.arraycopy(fileData, start, reqFileData, 0, copylen);
      client().atomicWriteFileStream(
        device.getAddress(),
        objectId,
        start,
        reqFileData);
      start += copylen;
    } while (start < writeLength);
  }

  private static int getTagSize(int dataLength)
  {
    if (dataLength <= 4)
    {
      // Application tag contains the length
      return 1;
    }
    else if (dataLength <= 253)
    {
      // Application tag + byte with length value
      return 2;
    }
    else if (dataLength <= 65535)
    {
      // Application tag + "254" + 2 bytes with length value
      return 4;
    }
    else
    {
      // Application tag + "255" + 4 bytes with length value
      return 6;
    }
  }


////////////////////////////////////////////////////////////////
// ReadFileReq
////////////////////////////////////////////////////////////////

  static class ReadFileReq
    implements Runnable
  {
    ReadFileReq(BReadFileConfig arg, BBacnetFile f)
    {
      parms = arg;
      bacnetFile = f;
    }

    public void run()
    {
      BBacnetDevice device = bacnetFile.device();
      if (device == null)
      {
        throw new IllegalStateException("Unable to read file because device not found: " + bacnetFile.getFileOrd());
      }

      int start = parms.getStart();
      int count = parms.getCount();
      byte[] fileData;

      try
      {
        int fileAccessMethod = AsnUtil.fromAsnEnumerated(
          client().readProperty(
            device.getAddress(),
            bacnetFile.getObjectId(),
            BBacnetPropertyIdentifier.FILE_ACCESS_METHOD));

        if (fileAccessMethod == BBacnetFileAccessMethod.STREAM_ACCESS)
        {
          int fileSize = -1;
          try
          {
            // Stream access: just read the file & return it.
            fileSize = AsnUtil.fromAsnUnsignedInt(
              client().readProperty(
                device.getAddress(),
                bacnetFile.getObjectId(),
                BBacnetPropertyIdentifier.FILE_SIZE));
          }
          catch (ErrorException e)
          {
            if (e.getErrorType().getErrorCode() != BBacnetErrorCode.UNKNOWN_FILE_SIZE)
            {
              throw e;
            }
          }

          fileData = readFileDataStream(device, bacnetFile.getObjectId(), fileSize, start, count);
        }
        else
        {
          // Record access: read the file records, then put them together into
          // one big byte array.  The byte array will have to be ASN-decoded to
          // distinguish the individual records.
          fileData = readFileDataRecord(device, bacnetFile.getObjectId(), start, count);
        }
      }
      catch (BacnetException e)
      {
        log.log(Level.SEVERE, "Unable to read file contents for " + bacnetFile.getObjectId() + " : " + e, e);
        throw new BajaRuntimeException(e);
      }

      // Now write to our local file.
      RandomAccessFile out = null;
      try
      {
        if (bacnetFile.file == null)
          throw new NullOrdException("No local target file specified for BACnet File " + bacnetFile);
        if (bacnetFile.file.isReadonly())
          throw new IllegalStateException("Unable to write to file " + bacnetFile.getFileOrd());
        File f = ((BLocalFileStore)bacnetFile.file.getStore()).getLocalFile();
        out = new RandomAccessFile(f, "rw");

        out.write(fileData);
      }
      catch (IOException e)
      {
        log.log(Level.SEVERE, "IOException writing to local file " + bacnetFile.file, e);
        throw new BajaRuntimeException(e);
      }
      finally
      {
        if (out != null) try
        {
          out.close();
        }
        catch (IOException ignore)
        {
        }
      }
    }

    BReadFileConfig parms;
    BBacnetFile bacnetFile;
  }


////////////////////////////////////////////////////////////////
//ReadFileReq
////////////////////////////////////////////////////////////////

  class WriteFileReq
    implements Runnable
  {
    WriteFileReq(BWriteFileConfig arg, BBacnetFile f)
    {
      parms = arg;
      bacnetFile = f;
    }

    public void run()
    {
      BBacnetDevice device = bacnetFile.device();
      if (device == null)
      {
        throw new IllegalStateException("Unable to write to file because device not found: " + bacnetFile.getFileOrd());
      }

      int remoteStart = parms.getRemoteStart();
      int localStart = parms.getLocalStart();
      byte[] fileData;

      // First read the data from our local file.
      RandomAccessFile src = null;
      try
      {
        if (bacnetFile.file == null)
          throw new NullOrdException("No local source file specified for BACnet File " + bacnetFile);
        File f = ((BLocalFileStore)bacnetFile.file.getStore()).getLocalFile();
        long flen = f.length() - localStart;
        if (flen > Integer.MAX_VALUE)
          throw new BajaRuntimeException("Local file data length " + flen + " is too long to write to BACnet!");
        int len = (int)flen;
        fileData = new byte[len];
        src = new RandomAccessFile(f, "r");
        src.seek(localStart);

        src.read(fileData, 0, len);
      }
      catch (IOException e)
      {
        log.log(Level.SEVERE, "IOException reading from local file " + bacnetFile.file, e);
        throw new BajaRuntimeException(e);
      }
      finally
      {
        if (src != null) try
        {
          src.close();
        }
        catch (IOException ignore)
        {
        }
      }

      // Now write it to the remote file.
      try
      {
        int fileAccessMethod = AsnUtil.fromAsnEnumerated(
          client().readProperty(device.getAddress(),
            bacnetFile.getObjectId(),
            BBacnetPropertyIdentifier.FILE_ACCESS_METHOD));

        // Stream access: just read the file & return it.
        if (fileAccessMethod == BBacnetFileAccessMethod.STREAM_ACCESS)
        {
          writeFileDataStream(device,
            bacnetFile.getObjectId(),
            remoteStart,
            fileData);
        }
        else
        {
          try
          {
            BBacnetOctetString[] fileRecordData = getFileRecordData(fileData);
            writeFileDataRecord(device, getObjectId(), remoteStart, fileRecordData.length, fileRecordData);
          }
          catch (AsnException e)
          {
            log.severe("File data is not in array of encoded BACnetOctetStrings");
            throw new BajaRuntimeException(e);
          }
        }
      }
      catch (BacnetException e)
      {
        log.log(Level.SEVERE, "Unable to write file record contents for " + getObjectId() + ": " + e, e);
        throw new BajaRuntimeException(e);
      }

    }

    BWriteFileConfig parms;
    BBacnetFile bacnetFile;
  }


////////////////////////////////////////////////////////////////
//  Constants
////////////////////////////////////////////////////////////////

  // BACnet-Confirmed-Request-PDU header
  // [1] pdu-type (0), segmented-message, more-follows, segmented-response-accepted
  // [1] reserved, max-segments-accepted, max-apdu-length-accepted
  // [1] invoke-id
  // [0] sequence-number (optional): not included because the file data size will be limited to fit
  //     in a single segment
  // [0] proposed-window-size (optional): not included
  // [1] service-choice

  // Record access atomic write file requests:
  // [5] file-identifier (BACnetObjectIdentifier)
  // [1] choice (0) opening tag
  // [5] file-start-position (Integer): -1 (2 bytes) means append, otherwise value can be, at most,
  //     Integer.MAX_VALUE (5 bytes) in Java.
  // [?] file-data (OctetString): the size of the application tag depends on the length of the
  //     string
  // [1] choice closing tag
  private static final int STREAM_ACCESS_HEADER_SIZE = 16;

  // Record access atomic write file requests:
  // [5] file-identifier (BACnetObjectIdentifier)
  // [1] choice (1) opening tag
  // [5] file-start-record (Integer): -1 (2 bytes) means append, otherwise value can be, at most,
  //     Integer.MAX_VALUE (5 bytes) in Java.
  // [5] record-count (Unsigned): 0 (2 bytes) to Integer.MAX_VALUE (5 bytes) in Java
  // [?] file-record-data (SEQUENCE OF OctetString): the size of the application tag of each octet
  //     string depends on the length of the string
  // [1] choice closing tag
  private static final int RECORD_ACCESS_HEADER_SIZE = 21;

  // APDU size safety factor
  private static final int FILE_DATA_SAFETY_FACTOR = 10;

////////////////////////////////////////////////////////////////
//  Attributes
////////////////////////////////////////////////////////////////

  private BIFile file;

}
