/*
 * Copyright 2003 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static javax.baja.bacnet.enums.BBacnetErrorCode.datatypeNotSupported;

import javax.baja.bacnet.BBacnetNetwork;
import javax.baja.bacnet.BacnetConst;
import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetAddress;
import javax.baja.bacnet.datatypes.BBacnetDeviceObjectPropertyReference;
import javax.baja.bacnet.enums.BBacnetErrorClass;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.ErrorType;
import javax.baja.bacnet.io.PropertyValue;
import javax.baja.nre.annotations.AgentOn;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.schedule.BControlSchedule;
import javax.baja.schedule.BNumericSchedule;
import javax.baja.schedule.BWeeklySchedule;
import javax.baja.security.PermissionException;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusBoolean;
import javax.baja.status.BStatusEnum;
import javax.baja.status.BStatusNumeric;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BDynamicEnum;
import javax.baja.sys.BEnum;
import javax.baja.sys.BEnumRange;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.bacnet.asn.AsnInputStream;
import com.tridium.bacnet.asn.AsnUtil;
import com.tridium.bacnet.asn.NBacnetPropertyValue;
import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.asn.NReadPropertyResult;
import com.tridium.bacnet.stack.DeviceRegistry;

/**
 * BBacnetNumericScheduleDescriptor exposes a Niagara schedule to Bacnet.
 *
 * @author Craig Gemmill on 18 Aug 03
 * @since Niagara 3 Bacnet 1.0
 */
@NiagaraType(
  agent = @AgentOn(
    types = "schedule:NumericSchedule"
  )
)
/*
 @since Niagara 4.14u3
 @since Niagara 4.15u2
 */
@NiagaraProperty(
  name = "scheduleDataType",
  type = "BEnum",
  defaultValue = "BDynamicEnum.make(DATA_TYPE_REAL, NUMERIC_DATA_TYPE_RANGE)",
  flags = Flags.HIDDEN
)
public class BBacnetNumericScheduleDescriptor
  extends BBacnetScheduleDescriptor
{
  public static final int DATA_TYPE_UNSIGNED = 0;
  public static final int DATA_TYPE_INTEGER = 1;
  public static final int DATA_TYPE_REAL = 2;
  public static final int DATA_TYPE_DOUBLE = 3;
  public static final int DATA_TYPE_ENUMERATED = 4;

  public static final BEnumRange NUMERIC_DATA_TYPE_RANGE = BEnumRange.make(new String[] {
    AsnUtil.getAsnTypeName(BacnetConst.ASN_UNSIGNED),
    AsnUtil.getAsnTypeName(BacnetConst.ASN_INTEGER),
    AsnUtil.getAsnTypeName(BacnetConst.ASN_REAL),
    AsnUtil.getAsnTypeName(BacnetConst.ASN_DOUBLE),
    AsnUtil.getAsnTypeName(BacnetConst.ASN_ENUMERATED),
  });

//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BBacnetNumericScheduleDescriptor(3670233319)1.0$ @*/
/* Generated Mon May 12 08:56:07 CDT 2025 by Slot-o-Matic (c) Tridium, Inc. 2012-2025 */

  //region Property "scheduleDataType"

  /**
   * Slot for the {@code scheduleDataType} property.
   * @since Niagara 4.14u3
   * @since Niagara 4.15u2
   * @see #getScheduleDataType
   * @see #setScheduleDataType
   */
  @Generated
  public static final Property scheduleDataType = newProperty(Flags.HIDDEN, BDynamicEnum.make(DATA_TYPE_REAL, NUMERIC_DATA_TYPE_RANGE), null);

  /**
   * Get the {@code scheduleDataType} property.
   * @since Niagara 4.14u3
   * @since Niagara 4.15u2
   * @see #scheduleDataType
   */
  @Generated
  public BEnum getScheduleDataType() { return (BEnum)get(scheduleDataType); }

  /**
   * Set the {@code scheduleDataType} property.
   * @since Niagara 4.14u3
   * @since Niagara 4.15u2
   * @see #scheduleDataType
   */
  @Generated
  public void setScheduleDataType(BEnum v) { set(scheduleDataType, v, null); }

  //endregion Property "scheduleDataType"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetNumericScheduleDescriptor.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Constructor.
   */
  public BBacnetNumericScheduleDescriptor()
  {
  }

////////////////////////////////////////////////////////////////
// BComponent
////////////////////////////////////////////////////////////////

  @Override
  public void started()
    throws Exception
  {
    super.started();
    setScheduleDataType(BDynamicEnum.make(getScheduleDataType().getOrdinal(), NUMERIC_DATA_TYPE_RANGE));

    // On startup, unhide the new frozen property that was added in Niagara 4.14u3, 4.15u2.
    // Also set the USER_DEFINED_1 flag so that we don't do this again (in case the user wants to
    // re-hide this property)
    if (!Flags.isUserDefined1(this, scheduleDataType))
    {
      setFlags(scheduleDataType, getFlags(scheduleDataType) & ~Flags.HIDDEN | Flags.USER_DEFINED_1);
    }
  }

////////////////////////////////////////////////////////////////
// Actions
////////////////////////////////////////////////////////////////

  /**
   * Write the present value of the schedule to non-Present_Value
   * target properties, and to any external targets.
   */
  @Override
  public void doWritePresentValue()
  {
    BNumericSchedule schedule = (BNumericSchedule) getSchedule();
    if (schedule == null || !schedule.getEffective().isEffective(BAbsTime.now()))
    {
      return;
    }

    BStatusNumeric out = schedule.getOut();
    byte[] propertyValue = encodeAsn(out);

    for (BBacnetDeviceObjectPropertyReference ref : getListOfObjectPropertyReferences().getChildren(BBacnetDeviceObjectPropertyReference.class))
    {
      if (isRemoteDevice(ref))
      {
        writeToRemote(ref, propertyValue);
      }
      else
      {
        writeToLocal(ref, propertyValue);
      }
    }

    setLastEffectiveValue((BStatusValue) out.newCopy());
  }

  private byte[] encodeAsn(BStatusNumeric out)
  {
    if (out.getStatus().isNull())
    {
      return ASN_NULL_VALUE;
    }

    switch (getScheduleDataType().getOrdinal())
    {
      case DATA_TYPE_UNSIGNED:
        return AsnUtil.toAsnUnsigned((long) out.getValue());
      case DATA_TYPE_INTEGER:
        return AsnUtil.toAsnInteger((int) out.getValue());
      case DATA_TYPE_REAL:
        return AsnUtil.toAsnReal(out.getValue());
      case DATA_TYPE_DOUBLE:
        return AsnUtil.toAsnDouble(out.getValue());
      case DATA_TYPE_ENUMERATED:
        return AsnUtil.toAsnEnumerated((int) out.getValue());
      default:
        throw new IllegalStateException("Invalid Schedule Data Type for " + this + ":" + getScheduleDataType().getOrdinal());
    }
  }

  private static boolean isRemoteDevice(BBacnetDeviceObjectPropertyReference ref)
  {
    return ref.isDeviceIdUsed() && !(ref.getDeviceId().equals(BBacnetNetwork.localDevice().getObjectId()));
  }

  private void writeToRemote(BBacnetDeviceObjectPropertyReference ref, byte[] propertyValue)
  {
    BBacnetAddress addr = DeviceRegistry.getDeviceAddress(ref.getDeviceId());
    if (addr == null)
    {
      findOrAddRemoteDeviceAndPoint(ref);
      addr = DeviceRegistry.getDeviceAddress(ref.getDeviceId());
    }

    if (addr == null)
    {
      log.warning(this + ": Unable to write Schedule output to " + ref + ": unable to resolve device address");
      return;
    }

    try
    {
      client().writeProperty(
        addr,
        ref.getObjectId(),
        ref.getPropertyId(),
        ref.getPropertyArrayIndex(),
        propertyValue,
        getPriorityForWriting());
    }
    catch (Exception e)
    {
      log.warning(this + ": BacnetException writing schedule output from " + this + " to remote object " + ref + ": " + e);
    }
  }

  private void writeToLocal(BBacnetDeviceObjectPropertyReference ref, byte[] propertyValue)
  {
    BIBacnetExportObject export = BBacnetNetwork.localDevice().lookupBacnetObject(ref.getObjectId());
    try
    {
      ErrorType err = export.writeProperty(
        new NBacnetPropertyValue(
          ref.getPropertyId(),
          ref.getPropertyArrayIndex(),
          propertyValue,
          getPriorityForWriting()));
      if (err != null)
      {
        log.warning(this + ": Unable to write schedule output from " + this + " to local object " + ref + "; error: " + err);
      }
    }
    catch (Exception e)
    {
      log.warning(this + ": Unable to write schedule output from " + this + " to local object " + ref + ": " + e);
    }
  }

////////////////////////////////////////////////////////////////
// Overrides
////////////////////////////////////////////////////////////////

  /**
   * Override point for BBacnetScheduleDescriptors to enforce
   * type rules for their exposed schedules.
   *
   * @param sched the exposed schedule
   * @return true if the Niagara schedule type is legal for this schedule type.
   */
  @Override
  final boolean isScheduleTypeLegal(BWeeklySchedule sched)
  {
    return sched instanceof BNumericSchedule;
  }

  @Override
  protected boolean isEqual(int ansTypeOfRefObj, int asnTypeOfSchedule)
  {
    if (ansTypeOfRefObj == asnTypeOfSchedule)
    {
      return true;
    }

    switch (ansTypeOfRefObj)
    {
      case ASN_UNSIGNED:
        setScheduleDataType(BDynamicEnum.make(DATA_TYPE_UNSIGNED, NUMERIC_DATA_TYPE_RANGE));
        return true;
      case ASN_INTEGER:
        setScheduleDataType(BDynamicEnum.make(DATA_TYPE_INTEGER, NUMERIC_DATA_TYPE_RANGE));
        return true;
      case ASN_REAL:
        setScheduleDataType(BDynamicEnum.make(DATA_TYPE_REAL, NUMERIC_DATA_TYPE_RANGE));
        return true;
      case ASN_DOUBLE:
        setScheduleDataType(BDynamicEnum.make(DATA_TYPE_DOUBLE, NUMERIC_DATA_TYPE_RANGE));
        return true;
      case ASN_ENUMERATED:
        setScheduleDataType(BDynamicEnum.make(DATA_TYPE_ENUMERATED, NUMERIC_DATA_TYPE_RANGE));
        return true;
      default:
        return false;
    }
  }

  /**
   * Get the ASN type to use in encoding the TimeValues for this schedule.
   */
  @Override
  int getAsnType()
  {
    switch (getScheduleDataType().getOrdinal())
    {
      case DATA_TYPE_UNSIGNED:
        return ASN_UNSIGNED;
      case DATA_TYPE_INTEGER:
        return ASN_INTEGER;
      case DATA_TYPE_REAL:
        return ASN_REAL;
      case DATA_TYPE_DOUBLE:
        return ASN_DOUBLE;
      case DATA_TYPE_ENUMERATED:
        return ASN_ENUMERATED;
      default:
        throw new IllegalStateException("Invalid Schedule Data Type for " + this + ":" + getScheduleDataType().getOrdinal());
    }
  }

  /**
   * Get the output property to which we link.
   *
   * @return the output property for this schedule.
   */
  @Override
  final Property getScheduleOutputProperty()
  {
    return BNumericSchedule.out;
  }

  /**
   * Get the value of a property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @return a PropertyValue containing either the encoded value or the error.
   */
  @Override
  protected PropertyValue readProperty(int pId, int ndx)
  {
    BNumericSchedule sched = (BNumericSchedule)getSchedule();
    if (sched == null)
    {
      return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.OBJECT,
                                                              BBacnetErrorCode.TARGET_NOT_CONFIGURED));
    }

    // Check for array index on non-array property.
    if (ndx >= 0)
    {
      if (!isArray(pId))
      {
        return new NReadPropertyResult(pId, ndx, new NErrorType(BBacnetErrorClass.PROPERTY,
                                                                BBacnetErrorCode.PROPERTY_IS_NOT_AN_ARRAY));
      }
    }

    switch (pId)
    {
      case BBacnetPropertyIdentifier.PRESENT_VALUE:
        BStatusNumeric out;
        BAbsTime currentTime = BAbsTime.now();
        if (!sched.isEffective(currentTime) && getLastEffectiveValue() != null)
        {
          out = (BStatusNumeric) getLastEffectiveValue();
        }
        else
        {
          out = sched.getOut();
        }

        return new NReadPropertyResult(pId, ndx, encodeAsn(out));

      case BBacnetPropertyIdentifier.SCHEDULE_DEFAULT:
        BStatusNumeric sf = (BStatusNumeric)sched.getDefaultOutput();
        return new NReadPropertyResult(pId, ndx, encodeAsn(sf));

      default:
        return super.readProperty(pId, ndx);
    }
  }

  /**
   * Set the value of a property.
   * Subclasses with additional properties override this to check for
   * their properties.  If no match is found, call this superclass
   * method to check these properties.
   *
   * @param pId the requested property-identifier.
   * @param ndx the property array index (-1 if not specified).
   * @param val the Asn-encoded value for the property.
   * @param pri the priority level (only used for commandable properties).
   * @return null if everything goes OK, or
   * an ErrorType describing the error if not.
   */
  @Override
  protected ErrorType writeProperty(int pId,
                                    int ndx,
                                    byte[] val,
                                    int pri)
    throws BacnetException
  {
    BNumericSchedule sched = (BNumericSchedule)getSchedule();
    if (sched == null)
    {
      return new NErrorType(BBacnetErrorClass.OBJECT,
                            BBacnetErrorCode.TARGET_NOT_CONFIGURED);
    }

    // Check for array index on non-array property.
    if (ndx >= 0)
    {
      if (!isArray(pId))
      {
        return new NErrorType(BBacnetErrorClass.PROPERTY,
                              BBacnetErrorCode.PROPERTY_IS_NOT_AN_ARRAY);
      }
    }

    try
    {
      synchronized (asnIn)
      {
        asnIn.setBuffer(val);
        switch (pId)
        {
          case BBacnetPropertyIdentifier.PRESENT_VALUE:
            if (((BStatusValue)sched.get("out")).getStatus().isDisabled())
            {
              int tag = asnIn.peekApplicationTag();
              if (tag == ASN_NULL)
              {
                // If I set the input null, the schedule's schedule will override.
                // Set the input to non-null to allow me to override, and then
                // set the output null.
                // FIXX: This should be done with an atomic transaction, which
                //       does not currently exist in the framework.
                sched.getIn().set(
                  BStatusValue.status,
                  BStatus.make(sched.getIn().getStatus(), BStatus.NULL, false),
                  BLocalBacnetDevice.getBacnetContext());
                sched.getOut().set(
                  BStatusValue.status,
                  BStatus.make(sched.getOut().getStatus(), BStatus.NULL, true),
                  BLocalBacnetDevice.getBacnetContext());
                return null;
              }
              else
              {
                return updateProperty(sched, BNumericSchedule.in, asnIn, tag);
              }
            }
            else
              return new NErrorType(BBacnetErrorClass.PROPERTY,
                                    BBacnetErrorCode.WRITE_ACCESS_DENIED);

          default:
            return super.writeProperty(pId, ndx, val, pri);
        }
      }
    }
    catch (AsnException e)
    {
      log.warning("AsnException writing property " + pId + " in object " + getObjectId() + ": " + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
                            BBacnetErrorCode.INVALID_DATA_TYPE);
    }
    catch (PermissionException e)
    {
      log.warning("PermissionException writing property " + pId + " in object " + getObjectId() + ": " + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
                            BBacnetErrorCode.WRITE_ACCESS_DENIED);
    }
    catch (Exception e)
    {
      log.warning("Exception writing property " + pId + " in object " + getObjectId() + ": " + e);
      return new NErrorType(BBacnetErrorClass.PROPERTY,
                            BBacnetErrorCode.OTHER);
    }
  }

  private ErrorType updateProperty(BNumericSchedule sched, Property property, AsnInputStream asnIn, int tag)
    throws AsnException
  {
    int scheduleDataType = getScheduleDataType().getOrdinal();
    switch (tag)
    {
      case ASN_UNSIGNED:
        if (scheduleDataType == DATA_TYPE_UNSIGNED)
        {
          updateProperty(sched, property, asnIn.readUnsignedInteger());
          return null;
        }
        break;

      case ASN_INTEGER:
        if (scheduleDataType == DATA_TYPE_INTEGER)
        {
          updateProperty(sched, property, asnIn.readSignedInteger());
          return null;
        }
        break;

      case ASN_REAL:
        if (scheduleDataType == DATA_TYPE_REAL)
        {
          updateProperty(sched, property, asnIn.readReal());
          return null;
        }
        break;

      case ASN_DOUBLE:
        if (scheduleDataType == DATA_TYPE_DOUBLE)
        {
          updateProperty(sched, property, asnIn.readDouble());
          return null;
        }
        break;

      case ASN_ENUMERATED:
        if (scheduleDataType == DATA_TYPE_ENUMERATED)
        {
          updateProperty(sched, property, asnIn.readEnumerated());
          return null;
        }
        break;
    }

    // If we haven't returned yet, we are out of service, but the data type didn't match.
    return new NErrorType(BBacnetErrorClass.property, datatypeNotSupported);
  }

  private static void updateProperty(BNumericSchedule sched, Property property, double value)
  {
    BStatusNumeric statusNumeric = (BStatusNumeric) sched.get(property).newCopy();
    statusNumeric.setValue(value);
    statusNumeric.setStatusNull(false);
    sched.set(property, statusNumeric, BLocalBacnetDevice.getBacnetContext());
  }

  /**
   * Translate the status value to something appropriate for the changed type
   * @param statusValue
   * @return
   */
  @Override
  BStatusValue getEffectiveValueFrom(BStatusValue statusValue)
  {
    BStatusNumeric ret = new BStatusNumeric(0.0, BStatus.nullStatus);

    if (statusValue instanceof BStatusEnum)
    {
      ret.setValue(((BStatusEnum) statusValue).getValue().getOrdinal());
    }
    else if(statusValue instanceof BStatusBoolean)
    {
      ret.setValue(((BStatusBoolean)statusValue).getValue() ? 1 : 0);
    }

    return ret;
  }

  /**
   * Write the schedule default value for numeric type schedule
   * @param asnInputStream
   * @return null if no error, otherwise error code
   */
  @Override
  protected ErrorType doWriteScheduleDefaultValue(AsnInputStream asnInputStream, int applicationTag)
    throws Exception
  {
    BNumericSchedule sched = (BNumericSchedule) getSchedule();
    return updateProperty(sched, BControlSchedule.defaultOutput, asnInputStream, applicationTag);
  }
}
