/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.atomic.AtomicReference;

import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.control.BBooleanWritable;
import javax.baja.control.BControlPoint;
import javax.baja.control.BEnumWritable;
import javax.baja.control.BIWritablePoint;
import javax.baja.control.BNumericWritable;
import javax.baja.control.BPointExtension;
import javax.baja.control.BStringWritable;
import javax.baja.control.enums.BPriorityLevel;
import javax.baja.control.ext.BAbstractProxyExt;
import javax.baja.driver.point.BProxyExt;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.status.BStatus;
import javax.baja.status.BStatusBoolean;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BLink;
import javax.baja.sys.BValue;
import javax.baja.sys.BasicContext;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

@NiagaraType
@NiagaraProperty(
  name = "outOfService",
  type = "boolean",
  defaultValue = "false"
)
@NiagaraProperty(
  name = "presentValue",
  type = "BValue",
  defaultValue = "BBoolean.FALSE"
)
public class BOutOfServiceExt
  extends BPointExtension
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.export.BOutOfServiceExt(3606498683)1.0$ @*/
/* Generated Thu Jun 02 14:30:01 EDT 2022 by Slot-o-Matic (c) Tridium, Inc. 2012-2022 */

  //region Property "outOfService"

  /**
   * Slot for the {@code outOfService} property.
   * @see #getOutOfService
   * @see #setOutOfService
   */
  @Generated
  public static final Property outOfService = newProperty(0, false, null);

  /**
   * Get the {@code outOfService} property.
   * @see #outOfService
   */
  @Generated
  public boolean getOutOfService() { return getBoolean(outOfService); }

  /**
   * Set the {@code outOfService} property.
   * @see #outOfService
   */
  @Generated
  public void setOutOfService(boolean v) { setBoolean(outOfService, v, null); }

  //endregion Property "outOfService"

  //region Property "presentValue"

  /**
   * Slot for the {@code presentValue} property.
   * @see #getPresentValue
   * @see #setPresentValue
   */
  @Generated
  public static final Property presentValue = newProperty(0, BBoolean.FALSE, null);

  /**
   * Get the {@code presentValue} property.
   * @see #presentValue
   */
  @Generated
  public BValue getPresentValue() { return get(presentValue); }

  /**
   * Set the {@code presentValue} property.
   * @see #presentValue
   */
  @Generated
  public void setPresentValue(BValue v) { set(presentValue, v, null); }

  //endregion Property "presentValue"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BOutOfServiceExt.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Constructors
////////////////////////////////////////////////////////////////

  public BOutOfServiceExt()
  {
  }

////////////////////////////////////////////////////////////////
// BComponent
////////////////////////////////////////////////////////////////

  @Override
  public void started()
    throws Exception
  {
    super.started();

    // Set bacnetValue with the right type and value.
    // No need to execute the point: outOfServiceExt.presentValue is identical to the point's out
    // value.  Also, BControlPoint will execute itself because this outOfServiceExt was added.
    set(presentValue, getParentPoint().getOutStatusValue().getValueValue(), noExecuteContext);

    // Set present value flag to readOnly for writable type descriptors, contains links for changing
    // out slot value. But this is required for non-writable descriptor, no links, only have present
    // value property for changing out slot value.
    int presentValueFlags = getFlags(presentValue);
    presentValueFlags |= Flags.READONLY;
    setFlags(presentValue, presentValueFlags);

    reorderToTop();
  }

  private void reorderToTop()
  {
    // Reordering OutOfServiceExt to top so alarm and history extensions act on its present
    // value instead of the proxyExt's readValue when Out of Service is true.
    getParentPoint().reorderToTop(getPropertyInParent());
  }

  @Override
  public void changed(Property p, Context cx)
  {
    super.changed(p, cx);
    if (!isRunning())
    {
      return;
    }

    if (p.equals(outOfService))
    {
      BControlPoint point = getParentPoint();
      setPresentValue(point.getOutStatusValue().getValueValue());

      if (getOutOfService())
      {
        // 135-2024 12.3.10 (Analog Output), 12.7.10 (Binary Output), 12.19.10 (Multi-State Output)
        // When Out_Of_Service is TRUE, changes to the Present_Value property are decoupled from
        // the physical output.
        disableProxyExt(point);
      }
      else
      {
        restoreProxyExt(point);
      }

      if (export instanceof BBacnetLoopDescriptor)
      {
        if (point.getType().toString().equals("kitControl:LoopPoint"))
        {
          if (getOutOfService())
          {
            // Decouple the present value from algorithm
            point.set("loopEnable", new BStatusBoolean(false));
          }
          else
          {
            // Explicitly loop is enabled when out of service is false
            point.set("loopEnable", new BStatusBoolean(true));
          }
        }
      }

      if (isValueDescriptor(export) && export instanceof BacnetWritableDescriptor)
      {
        // Input links on the point of a non-commandable value descriptor do not need to be disabled
        // because the input slots will be overwritten by the outOfServiceExt's present value
        // property.
        // Input links on the point of an output descriptor can be changed by software local to the
        // BACnet device.

        // 135-2024 12.4.9 (Analog Value), 12.8.9 (Binary Value), and 12.20.9 (Multi-State Value)
        // When Out_Of_Service is TRUE:
        // (a) the Present_Value of the object is prevented from being changed by software local to
        // the BACnet device in which the object resides;
        // (b) the Present_Value property ... shall be writable to allow simulating specific
        // conditions or for testing purposes;
        // ...
        // (d) if the Priority_Array and Relinquish_Default properties are present, the
        // Present_Value property shall still be controlled by the BACnet command prioritization
        // mechanism (see Clause 19).
        //
        // Restrictions on changing the Present_Value property by software local to the BACnet
        // device do not apply to local human-machine interfaces.
        if (getOutOfService())
        {
          disableInputLinks((BIWritablePoint) point, export);
        }
        else
        {
          restoreInputLinks((BIWritablePoint) point, export);
        }
      }

      if (export instanceof BBacnetEventSource)
      {
        BBacnetEventSource eventSource = (BBacnetEventSource)export;
        eventSource.checkValid();
        eventSource.statusChanged();
      }
    }

    if (cx != noExecuteContext)
    {
      executePoint();
      if (export instanceof BIBacnetCovSource)
      {
        ((BIBacnetCovSource)export).checkCov();
      }
    }
  }

  private void disableProxyExt(BControlPoint point)
  {
    BProxyExt proxyExt = getProxyExt(point);
    if (proxyExt != null)
    {
      wasProxyExtEnabled = proxyExt.getEnabled();
      proxyExt.setEnabled(false);
    }
  }

  private void restoreProxyExt(BControlPoint point)
  {
    BProxyExt proxyExt = getProxyExt(point);
    if (proxyExt != null && wasProxyExtEnabled)
    {
      proxyExt.setEnabled(true);
    }
  }

  private BProxyExt getProxyExt(BControlPoint point)
  {
    BAbstractProxyExt proxyExt = point.getProxyExt();
    if (proxyExt instanceof BProxyExt)
    {
      return (BProxyExt) proxyExt;
    }

    return null;
  }

  private static boolean isValueDescriptor(BIBacnetExportObject descriptor)
  {
    int type = descriptor.getObjectId().getObjectType();
    switch (type)
    {
      case BBacnetObjectType.ANALOG_VALUE:
      case BBacnetObjectType.BINARY_VALUE:
      case BBacnetObjectType.MULTI_STATE_VALUE:
        return true;
      default:
        return false;
    }
  }

  private void disableInputLinks(BIWritablePoint point, BIBacnetExportObject descriptor)
  {
    Map<BLink, Boolean> linkEnabledStates = new WeakHashMap<>();

    for (BPriorityLevel level : PRIORITY_LEVELS)
    {
      Property inProperty = point.getInProperty(level);
      BLink[] links = ((BControlPoint) point).getLinks(inProperty);
      for (BLink link : links)
      {
        // Do not disable links from the descriptor so that changes from the BACnet priority array
        // are still propagated.
        if (link.getSourceComponent() == descriptor)
        {
          continue;
        }

        linkEnabledStates.put(link, link.getEnabled());
        link.setEnabled(false);
      }
    }

    this.linkEnabledStates.set(linkEnabledStates);
  }

  private void restoreInputLinks(BIWritablePoint point, BIBacnetExportObject descriptor)
  {
    Map<BLink, Boolean> linkEnabledStates = this.linkEnabledStates.getAndSet(null);

    for (BPriorityLevel level : PRIORITY_LEVELS)
    {
      Property inProperty = point.getInProperty(level);
      BLink[] links = ((BControlPoint) point).getLinks(inProperty);
      for (BLink link : links)
      {
        // Do not restore links from the descriptor: they were not disabled as part setting
        // out-of-service to true and should not be enabled as part of setting it to false.
        if (link.getSourceComponent() == descriptor)
        {
          continue;
        }

        boolean wasEnabled = true;
        if (linkEnabledStates != null)
        {
          Boolean enabledState = linkEnabledStates.get(link);
          if (enabledState != null)
          {
            wasEnabled = enabledState;
          }
        }

        if (wasEnabled)
        {
          link.setEnabled(true);
          // Propagate in case the source changed while the link was disabled.
          if (link.getSourceSlot().isProperty() && link.getSourceSlot().isProperty())
          {
            link.propagate(null);
          }
        }
      }
    }
  }

  private static final BPriorityLevel[] PRIORITY_LEVELS =
  {
    BPriorityLevel.level_1,
    BPriorityLevel.level_2,
    BPriorityLevel.level_3,
    BPriorityLevel.level_4,
    BPriorityLevel.level_5,
    BPriorityLevel.level_6,
    BPriorityLevel.level_7,
    BPriorityLevel.level_8,
    BPriorityLevel.level_9,
    BPriorityLevel.level_10,
    BPriorityLevel.level_11,
    BPriorityLevel.level_12,
    BPriorityLevel.level_13,
    BPriorityLevel.level_14,
    BPriorityLevel.level_15,
    BPriorityLevel.level_16
  };

////////////////////////////////////////////////////////////////
// BPointExtension
////////////////////////////////////////////////////////////////

  @Override
  public void onExecute(BStatusValue working, Context cx)
  {
    // If the extension is "out of service", drive the working value
    // to the BACnet-written value.
    if (getOutOfService())
    {
      BControlPoint point = getParentPoint();
      BProxyExt proxyExt = getProxyExt(point);

      if (isCommandable)
      {
        if (proxyExt != null)
        {
          proxyExt.writeReset();
        }
      }

      if (export instanceof BacnetWritableDescriptor)
      {
        // outOfServiceExt's present value property is readOnly for commandable objects and does not
        // drive the working value.  Instead, the working value is driven by the input slots.  All
        // links to the input slots not from the descriptor have been disabled so only values from
        // BACnet writes have an effect.

        if (proxyExt != null)
        {
          // For commandable objects with non-null proxyExts, we need to overwrite the
          // proxyExt's readValue with a value based on the input slots.  Output objects that should
          // have a non-null proxyExt should be decoupled from the physical output.
          //
          // 135-2024 12.3.10 (Analog Output), 12.7.10 (Binary Output), 12.19.10 (Multi-State Output)
          // When Out_Of_Service is TRUE, changes to the Present_Value property are decoupled from
          // the physical output.
          BStatusValue inValue = getInValue((BIWritablePoint) getParentPoint());
          working.setStatus(inValue.getStatus());
          working.setValueValue(inValue.getValueValue());
        }
      }
      else
      {
        // Overwrite the working value with the outOfServiceExt's present value.  For input types,
        // the underlying point should be a non-writable point with a non-null proxyExt.  This will
        // overwrite the proxyExt's readValue.  For non-commandable value types, the underlying
        // point could be writable or non-writable and the proxyExt could be null or non-null.
        // Regardless, the outOfServiceExt's present value will overwrite the active value (if
        // writable) and the proxyExt's readValue (if non-null).
        //
        // 135-2024 12.2.10 (Analog Input), 12.6.10 (Binary Input), 12.18.10 (Multi-State Input)
        // When Out_Of_Service is TRUE, the Present_Value property is decoupled from the physical
        // input and will not track changes to the physical input.
        working.setValueValue(getPresentValue());
      }

      // Indicate on the point that something is unusual. Since the proxy ext is disabled, the point
      // status would be disabled and that may interrupt alarm or history exts. Thus, we are
      // changing the Niagara status to "overridden" makes.  This must be fixed, however, when
      // reporting BACnet status.
      BStatus status = working.getStatus();
      status = BStatus.makeDisabled(status, false);
      status = BStatus.makeOverridden(status, true);
      working.setStatus(status);
    }
  }

  private BStatusValue getInValue(BIWritablePoint point)
  {
    // Please avoid using BIWritablePoint#getActiveLevel
    BPriorityLevel activeLevel = getActiveLevel(point);
    if (activeLevel == BPriorityLevel.fallback)
    {
      if (point instanceof BBooleanWritable)
      {
        return ((BBooleanWritable) point).getFallback();
      }
      else if (point instanceof BNumericWritable)
      {
        return ((BNumericWritable) point).getFallback();
      }
      else if (point instanceof BEnumWritable)
      {
        return ((BEnumWritable) point).getFallback();
      }
      else
      {
        return ((BStringWritable) point).getFallback();
      }
    }
    else
    {
      return point.getInStatusValue(activeLevel);
    }
  }

  private BPriorityLevel getActiveLevel(BIWritablePoint point)
  {
    int activeLevel = 17;
    for(int level=1; level<=16; ++level)
    {
      BStatusValue in = point.getInStatusValue(BPriorityLevel.make(level));
      if (in.getStatus().isValid())
      {
        activeLevel = level;
        break;
      }
    }

    return BPriorityLevel.make(activeLevel);
  }

////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  public BIBacnetExportObject getExport()
  {
    return export;
  }

  public void setExport(BIBacnetExportObject exp)
  {
    export = exp;
  }

  public void setCommandable(boolean commandable)
  {
    isCommandable = commandable;
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private static final Context noExecuteContext = new BasicContext();

  BIBacnetExportObject export;
  boolean isCommandable;

  // Whether the point's proxy ext was enabled when outOfService became true.
  private boolean wasProxyExtEnabled = true;
  // Whether the links to value object links were enabled when outOfService became true.
  private final AtomicReference<Map<BLink, Boolean>> linkEnabledStates = new AtomicReference<>();
}
