/*
 * Copyright 2018 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import javax.baja.bacnet.BBacnetDevice;
import javax.baja.bacnet.BBacnetNetwork;
import javax.baja.bacnet.BIBacnetObjectContainer;
import javax.baja.bacnet.BacnetConfirmedServiceChoice;
import javax.baja.bacnet.BacnetConst;
import javax.baja.bacnet.BacnetException;
import javax.baja.bacnet.datatypes.BBacnetDeviceObjectPropertyReference;
import javax.baja.bacnet.datatypes.BBacnetObjectIdentifier;
import javax.baja.bacnet.enums.BBacnetErrorClass;
import javax.baja.bacnet.enums.BBacnetErrorCode;
import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.bacnet.enums.BBacnetPropertyIdentifier;
import javax.baja.bacnet.export.extensions.BBacnetRemoteUnsignedPropertyExt;
import javax.baja.bacnet.export.extensions.BBacnetUnsignedPropertyExt;
import javax.baja.bacnet.io.ChangeListError;
import javax.baja.bacnet.point.BBacnetBooleanProxyExt;
import javax.baja.bacnet.point.BBacnetEnumProxyExt;
import javax.baja.bacnet.point.BBacnetNumericProxyExt;
import javax.baja.bacnet.point.BBacnetProxyExt;
import javax.baja.bacnet.point.BBacnetStringProxyExt;
import javax.baja.bacnet.util.PropertyInfo;
import javax.baja.control.BBooleanWritable;
import javax.baja.control.BControlPoint;
import javax.baja.control.BEnumWritable;
import javax.baja.control.BNumericWritable;
import javax.baja.control.BPointExtension;
import javax.baja.control.BStringWritable;
import javax.baja.control.ext.BDiscreteTotalizerExt;
import javax.baja.naming.BOrd;
import javax.baja.naming.SlotPath;
import javax.baja.status.BStatusNumeric;
import javax.baja.sys.BComponent;
import javax.baja.sys.BDynamicEnum;
import javax.baja.sys.BFacets;
import javax.baja.sys.BLink;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BValue;
import javax.baja.sys.Flags;
import javax.baja.sys.Knob;
import javax.baja.sys.Sys;
import javax.baja.util.BFolder;

import com.tridium.bacnet.asn.NErrorType;
import com.tridium.bacnet.history.BBacnetBitStringTrendLogExt;
import com.tridium.bacnet.history.BBacnetBitStringTrendLogRemoteExt;
import com.tridium.bacnet.history.BBacnetBooleanTrendLogExt;
import com.tridium.bacnet.history.BBacnetBooleanTrendLogRemoteExt;
import com.tridium.bacnet.history.BBacnetEnumTrendLogExt;
import com.tridium.bacnet.history.BBacnetEnumTrendLogRemoteExt;
import com.tridium.bacnet.history.BBacnetNumericTrendLogExt;
import com.tridium.bacnet.history.BBacnetNumericTrendLogRemoteExt;
import com.tridium.bacnet.history.BBacnetStringTrendLogExt;
import com.tridium.bacnet.history.BBacnetStringTrendLogRemoteExt;
import com.tridium.bacnet.history.BBacnetTrendLogRemoteExt;
import com.tridium.bacnet.history.BIBacnetTrendLogExt;
import com.tridium.bacnet.services.error.NChangeListError;
import com.tridium.bacnet.stack.server.BBacnetExportTable;

/**
 * A collection of methods for interacting with BacnetDescriptor instances.
 *
 * @author Sandipan Aich on 25/10/2018
 */
public final class BacnetDescriptorUtil
{
  // Prevent object instantiation, static utility only
  private BacnetDescriptorUtil()
  {
  }

  static boolean isValid(BBacnetDeviceObjectPropertyReference reference)
  {
    if (reference == null)
    {
      return false;
    }

    if (!BBacnetPropertyIdentifier.isValid(reference.getPropertyId()) ||
        !reference.getObjectId().isValid())
    {
      return false;
    }

    BBacnetObjectIdentifier deviceId = reference.getDeviceId();
    if (deviceId.getObjectType() != BBacnetObjectType.DEVICE)
    {
      return false;
    }

    int deviceNum = deviceId.getInstanceNumber();
    return deviceNum >= -1 && deviceNum < BBacnetObjectIdentifier.UNCONFIGURED_INSTANCE_NUMBER;
  }

  /**
   * Returns true if the supplied deviceNum is -1 or matches the local device instance.
   * @since Niagara 4.14u2
   * @since Niagara 4.15u1
   */
  public static boolean isLocalDevice(int deviceNum)
  {
    if (deviceNum == -1)
    {
      return true;
    }

    BLocalBacnetDevice localDevice = BBacnetNetwork.localDevice();
    int localDeviceNum = localDevice.getObjectId().getInstanceNumber();
    return localDeviceNum == deviceNum;
  }

  static BControlPoint findOrAddPoint(BBacnetDeviceObjectPropertyReference objectPropRef)
    throws Exception
  {
    int deviceNum = objectPropRef.getDeviceId().getInstanceNumber();
    if (isLocalDevice(deviceNum))
    {
      return findOrAddLocalPoint(
        objectPropRef.getObjectId(),
        objectPropRef.getPropertyId(),
        objectPropRef.getPropertyArrayIndex());
    }
    else
    {
      return findOrAddRemotePoint(objectPropRef);
    }
  }

  static BControlPoint findOrAddLocalPoint(BBacnetObjectIdentifier objectId, int propertyId, int propertyArrayIndex)
    throws Exception
  {
    BComponent point = findLocalObject(objectId);
    if (!(point instanceof BControlPoint))
    {
      return null;
    }

    if (propertyId == BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME)
    {
      point = getPointForElapsedActiveTime(objectId, propertyArrayIndex, (BControlPoint) point);
    }

    return (BControlPoint) point;
  }

  static BComponent findLocalObject(BBacnetObjectIdentifier objectId)
    throws Exception
  {
    BIBacnetExportObject exportObject = BBacnetNetwork.localDevice().lookupBacnetObject(objectId);
    if (exportObject == null)
    {
      throw new Exception("Could not find a local BACnet export object with ID " + objectId);
    }

    BOrd exportObjectOrd = exportObject.getObjectOrd();
    if (exportObjectOrd.isNull())
    {
      throw new Exception("ObjectOrd is null for local BACnet object with ID " + objectId);
    }

    return (BComponent) exportObjectOrd.get(Sys.getStation());
  }

  private static BControlPoint getPointForElapsedActiveTime(
    BBacnetObjectIdentifier objectId,
    int propertyIndex,
    BControlPoint point)
  {
    BDiscreteTotalizerExt[] extensions = point.getChildren(BDiscreteTotalizerExt.class);
    BDiscreteTotalizerExt extension;
    BControlPoint linkedPoint = null;
    if (extensions.length > 0)
    {
      //Use the first Discrete Totalizer for algorithmic reporting
      extension =  extensions[0];
      extension.setEaTimeUpdateInterval(BRelTime.make(1000));
      linkedPoint = getNumericPointLinkedToDiscreteTotExt(objectId, extension);
    }
    else
    {
      extension = addDiscreteTotalizerExtToPoint(point);
    }

    if (linkedPoint == null)
    {
      linkedPoint = addPropertyPoint(null, objectId, BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME, propertyIndex);
      linkToNumericPoint(extension, linkedPoint);
    }

    return linkedPoint;
  }

  private static void linkToNumericPoint(BDiscreteTotalizerExt extension, BControlPoint linkedPoint)
  {
    try
    {
      BControlPoint divide;
      BValue divideCheckVar = linkedPoint.get("divide");
      if (divideCheckVar == null)
      {
        Class<?> divideClass = Sys.loadClass("kitControl", "com.tridium.kitControl.math.BDivide");
        divide = (BControlPoint)divideClass.getDeclaredConstructor().newInstance();
        divide.set("inB", new BStatusNumeric(BacnetConst.THOUSAND));
        linkedPoint.add("divide", divide);
      }
      else
      {
        divide = (BControlPoint) divideCheckVar;
      }

      BLink linkToDivide = new BLink(extension.getHandleOrd(),
                                     BDiscreteTotalizerExt.elapsedActiveTimeNumeric.getName(),
                                     "inA", true);
      divide.add(null, linkToDivide, BLocalBacnetDevice.getBacnetContext());

      BLink link = new BLink(divide.getHandleOrd(),"out",
                             BNumericWritable.in16.getName(), true);

      linkedPoint.add(null, link, BLocalBacnetDevice.getBacnetContext());
    }
    catch (ClassNotFoundException e)
    {
      removePoint(linkedPoint);
      logger.severe("Class BDivide is not found or kitControl module is not found: " + e);
    }
    catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException | InstantiationException e)
    {
      removePoint(linkedPoint);
      logger.severe("Exception while creating instance of divide: " + e);
    }
  }

  private static BDiscreteTotalizerExt addDiscreteTotalizerExtToPoint(BComponent point)
  {
    BDiscreteTotalizerExt discreteTotalizerExt = new BDiscreteTotalizerExt();
    discreteTotalizerExt.setEaTimeUpdateInterval(BRelTime.make(1000));
    point.add(DISCRETE_TOTALIZER_EXT + "?", discreteTotalizerExt);
    return discreteTotalizerExt;
  }

  private static BControlPoint getNumericPointLinkedToDiscreteTotExt(BBacnetObjectIdentifier objectId, BDiscreteTotalizerExt extension)
  {
    Knob[] knobs = extension.getKnobs(BDiscreteTotalizerExt.elapsedActiveTimeNumeric);
    List<BControlPoint> points = new ArrayList<>();
    for (Knob knob : knobs)
    {
      BComponent targetParent = knob.getTargetComponent().getParent().getParentComponent();
      if (targetParent instanceof BControlPoint)
      {
        points.add((BControlPoint) targetParent);
      }
    }
    if (!points.isEmpty())
    {
      return findElapsedActiveTimePoint(points.toArray(EMPTY_POINT_ARRAY), objectId);
    }
    return null;
  }

  static BControlPoint findOrAddRemotePoint(BBacnetDeviceObjectPropertyReference objPropRef)
  {
    BBacnetDevice device = findOrAddRemoteDevice(objPropRef.getDeviceId());
    BBacnetObjectIdentifier objectId = objPropRef.getObjectId();
    int propertyId = objPropRef.getPropertyId();
    int propertyIndex = objPropRef.getPropertyArrayIndex();
    BControlPoint point = findRemotePoint(device, objectId, propertyId, propertyIndex);
    if (point == null)
    {
      point = addPropertyPoint(device, objectId, propertyId, propertyIndex);
    }
    return point;
  }

  private static BControlPoint findRemotePoint(
    BBacnetDevice device,
    BBacnetObjectIdentifier objectId,
    int propertyId,
    int propertyArrayIndex)
  {
    if (propertyId == BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME)
    {
      BControlPoint[] remotePoints = device.getPoints().getPoints();
      BControlPoint point = findElapsedActiveTimePoint(remotePoints, objectId);
      if (point != null)
      {
        return point;
      }
    }

    return (BControlPoint) device.lookupBacnetObject(objectId, propertyId, propertyArrayIndex, BIBacnetObjectContainer.POINT);
  }

  private static BControlPoint findElapsedActiveTimePoint(BControlPoint[] points, BBacnetObjectIdentifier objectId)
  {
    if (points == null)
    {
      return null;
    }

    for (BControlPoint point : points)
    {
      BBacnetUnsignedPropertyExt[] extensions = point.getChildren(BBacnetUnsignedPropertyExt.class);
      for (BBacnetUnsignedPropertyExt ext : extensions)
      {
        if (ext.getPropertyId() == BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME &&
            ext.getObjectId().equivalent(objectId))
        {
          return point;
        }
      }
    }

    return null;
  }

  private static BControlPoint addPropertyPoint(
    BBacnetDevice device,
    BBacnetObjectIdentifier objectId,
    int propertyId,
    int propertyArrayIndex)
  {
    try
    {
      BControlPoint point = makePropertyPoint(device, objectId, propertyId, propertyArrayIndex);
      String name = makePointName(objectId, propertyId, propertyArrayIndex);
      if (device == null)
      {
        BComponent dynamicPointsFolder = (BComponent) BBacnetNetwork.bacnet().get(BacnetConst.DYNAMIC_POINTS_CREATED_FOR_EVENT_ENROLLMENT);
        if (dynamicPointsFolder == null)
        {
          dynamicPointsFolder = new BFolder();
          BBacnetNetwork.bacnet().add(BacnetConst.DYNAMIC_POINTS_CREATED_FOR_EVENT_ENROLLMENT, dynamicPointsFolder, Flags.HIDDEN | Flags.READONLY);
        }

        dynamicPointsFolder.add(SlotPath.escape(name), point);
      }
      else
      {
        device.getPoints().add(SlotPath.escape(name), point);
      }

      return point;
    }
    catch (Exception e)
    {
      logger.severe("Could not add point for property " + propertyId + " on device " + device + "; " + e.getMessage());
      return null;
    }
  }

  /**
   * Based on how BBacnetPointManager.Learn#toRow(Object, MgrEditRow) sets the row's default name.
   *
   * @since Niagara 4.10u5
   * @since Niagara 4.12u2
   * @since Niagara 4.13
   */
  private static String makePointName(BBacnetObjectIdentifier objectId, int propertyId, int propertyArrayIndex)
  {
    StringBuilder pointName = new StringBuilder(objectId.toString());

    if (propertyId != BBacnetPropertyIdentifier.PRESENT_VALUE)
    {
      pointName.append('-').append(BBacnetPropertyIdentifier.tag(propertyId));
    }

    if (propertyArrayIndex > 0)
    {
      pointName.append('_').append(propertyArrayIndex);
    }

    return FORWARD_SLASH_PATTERN.matcher(pointName.toString()).replaceAll(".");
  }

  private static final Pattern FORWARD_SLASH_PATTERN = Pattern.compile("/");

  private static void removePoint(BControlPoint point)
  {
    if (null == BBacnetNetwork.bacnet().get(BacnetConst.DYNAMIC_POINTS_CREATED_FOR_EVENT_ENROLLMENT))
    {
      BBacnetNetwork.bacnet().get(BacnetConst.DYNAMIC_POINTS_CREATED_FOR_EVENT_ENROLLMENT).asComponent().remove(point.getName());
    }
  }

  private static BBacnetDevice findOrAddRemoteDevice(BBacnetObjectIdentifier deviceId)
  {
    BBacnetNetwork network = BBacnetNetwork.bacnet();
    BBacnetDevice device = network.lookupDeviceById(deviceId);
    if (device == null)
    {
      device = addRemoteDevice(deviceId.getInstanceNumber());
    }

    return device;
  }

  private static BBacnetDevice addRemoteDevice(int instanceNum)
  {
    BBacnetObjectIdentifier id = BBacnetObjectIdentifier.make(BBacnetObjectType.DEVICE, instanceNum);

    BBacnetDevice device = new BBacnetDevice();
    device.setObjectId(id, null);
    BBacnetNetwork.bacnet().add(null, device);
    return device;
  }

  private static BControlPoint addPointForElapsedActiveTime(BBacnetObjectIdentifier bOid, boolean isLocal)
  {
    BControlPoint point = new BNumericWritable();
    BPointExtension extension = null;
    if (isLocal)
    {
      extension = new BBacnetUnsignedPropertyExt(bOid, BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME);
    }
    else
    {
      extension = new BBacnetRemoteUnsignedPropertyExt(bOid, BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME);
    }
    point.add("ElapsedActiveTimeExtension?", extension);
    return point;
  }

  private static BControlPoint makePropertyPoint(
    BBacnetDevice device,
    BBacnetObjectIdentifier objectId,
    int propertyId,
    int propertyArrayIndex)
      throws BacnetException
  {
    if (propertyId == BBacnetPropertyIdentifier.ELAPSED_ACTIVE_TIME)
    {
      return addPointForElapsedActiveTime(objectId, device == null);
    }

    int objectType = objectId.getObjectType();
    PropertyInfo propInfo = device.getPropertyInfo(objectType, propertyId);
    if (propInfo == null)
    {
      throw new BacnetException(
        "BACnet property information not found when making a property point" +
        "; object type: " + BBacnetObjectType.tag(objectType) +
        ", property ID: " + BBacnetPropertyIdentifier.tag(propertyId));
    }

    BControlPoint point = makePointForPropertyInfo(objectType, propInfo);

    BBacnetProxyExt ext = (BBacnetProxyExt) point.getProxyExt();
    ext.setDeviceFacets((BFacets) point.getFacets().newCopy());
    ext.setDataType(propInfo.getDataType());
    ext.setObjectId(objectId);
    ext.setPropertyId(BDynamicEnum.make(BBacnetPropertyIdentifier.make(propertyId)));
    ext.setPropertyArrayIndex(propertyArrayIndex);
    ext.setEnabled(true);

    return point;
  }

  private static BControlPoint makePointForPropertyInfo(int objectType, PropertyInfo propInfo)
    throws BacnetException
  {
    switch (propInfo.getAsnType())
    {
      case BacnetConst.ASN_NULL:
        return makeBacnetStringWritable();

      case BacnetConst.ASN_BOOLEAN:
        return makeBacnetBooleanWritable();

      case BacnetConst.ASN_UNSIGNED:
        return isMultiStatePresentValue(propInfo.getId(), objectType) ?
          makeBacnetEnumWritable() :
          makeBacnetNumericWritable();

      case BacnetConst.ASN_INTEGER:
      case BacnetConst.ASN_REAL:
      case BacnetConst.ASN_DOUBLE:
        return makeBacnetNumericWritable();

      case BacnetConst.ASN_OCTET_STRING:
      case BacnetConst.ASN_CHARACTER_STRING:
      case BacnetConst.ASN_BIT_STRING:
        return makeBacnetStringWritable();

      case BacnetConst.ASN_ENUMERATED:
        return propInfo.getType().equals("bacnet:BacnetBinaryPv") ?
          makeBacnetBooleanWritable() :
          makeBacnetEnumWritable();

      case BacnetConst.ASN_DATE:
      case BacnetConst.ASN_TIME:
      case BacnetConst.ASN_OBJECT_IDENTIFIER:
      case BacnetConst.ASN_CONSTRUCTED_DATA:
      case BacnetConst.ASN_BACNET_ARRAY:
      case BacnetConst.ASN_BACNET_LIST:
      case BacnetConst.ASN_ANY:
      case BacnetConst.ASN_CHOICE:
      case BacnetConst.ASN_UNKNOWN_PROPRIETARY:
        return makeBacnetStringWritable();

      default:
        throw new BacnetException("BACnet property type " + BBacnetPropertyIdentifier.tag(objectType)
          + " is not supported when making a property point");
    }
  }

  private static BBooleanWritable makeBacnetBooleanWritable()
  {
    BBooleanWritable booleanWritable = new BBooleanWritable();
    booleanWritable.setProxyExt(new BBacnetBooleanProxyExt());
    return booleanWritable;
  }

  private static BNumericWritable makeBacnetNumericWritable()
  {
    BNumericWritable numericWritable = new BNumericWritable();
    numericWritable.setProxyExt(new BBacnetNumericProxyExt());
    return numericWritable;
  }

  private static BEnumWritable makeBacnetEnumWritable()
  {
    BEnumWritable enumWritable = new BEnumWritable();
    enumWritable.setProxyExt(new BBacnetEnumProxyExt());
    return enumWritable;
  }

  private static BStringWritable makeBacnetStringWritable()
  {
    BStringWritable stringWritable = new BStringWritable();
    stringWritable.setProxyExt(new BBacnetStringProxyExt());
    return stringWritable;
  }

  private static boolean isMultiStatePresentValue(int propertyId, int objectType)
  {
    return propertyId == BBacnetPropertyIdentifier.PRESENT_VALUE &&
           (objectType == BBacnetObjectType.MULTI_STATE_INPUT ||
            objectType == BBacnetObjectType.MULTI_STATE_OUTPUT ||
            objectType == BBacnetObjectType.MULTI_STATE_VALUE);
  }

  static boolean isEqual(BBacnetDeviceObjectPropertyReference dopr1 , BBacnetDeviceObjectPropertyReference dopr2)
  {
    if (dopr1 == null && dopr2 == null)
    {
      //Both are null values
      return true;
    }

    if (dopr1 == null || dopr2 == null)
    {
      //Only one is a null value, can't be equal
      return false;
    }

    if (dopr1.isNull() && dopr2.isNull())
    {
      //Both are logically null
      return true;
    }

    return dopr1.getDeviceId().getInstanceNumber() == dopr2.getDeviceId().getInstanceNumber() &&
           dopr1.getObjectId().getObjectType() == dopr2.getObjectId().getObjectType() &&
           dopr1.getObjectId().getInstanceNumber() == dopr2.getObjectId().getInstanceNumber() &&
           dopr1.getPropertyId() == dopr2.getPropertyId() &&
           dopr1.getPropertyArrayIndex() == dopr2.getPropertyArrayIndex();
  }

  static BIBacnetTrendLogExt makeTrendLogExt(
    int objectType,
    PropertyInfo propInfo,
    boolean isRemote)
      throws BacnetException
  {
    switch (propInfo.getAsnType())
    {
      case BacnetConst.ASN_BOOLEAN:
        return isRemote ?
          new BBacnetBooleanTrendLogRemoteExt() :
          new BBacnetBooleanTrendLogExt();

      case BacnetConst.ASN_UNSIGNED:
        if (isMultiStatePresentValue(propInfo.getId(), objectType))
        {
          return isRemote ?
            new BBacnetEnumTrendLogRemoteExt() :
            new BBacnetEnumTrendLogExt();
        }
        else
        {
          return isRemote ?
            new BBacnetNumericTrendLogRemoteExt() :
            new BBacnetNumericTrendLogExt();
        }

      case BacnetConst.ASN_INTEGER:
      case BacnetConst.ASN_REAL:
      case BacnetConst.ASN_DOUBLE:
        return isRemote ?
          new BBacnetNumericTrendLogRemoteExt() :
          new BBacnetNumericTrendLogExt();

      case BacnetConst.ASN_BIT_STRING:
        return isRemote ?
          new BBacnetBitStringTrendLogRemoteExt() :
          new BBacnetBitStringTrendLogExt();

      case BacnetConst.ASN_ENUMERATED:
        if (propInfo.getType().equals("bacnet:BacnetBinaryPv"))
        {
          return isRemote ?
            new BBacnetBooleanTrendLogRemoteExt() :
            new BBacnetBooleanTrendLogExt();
        }
        else
        {
          return isRemote ?
            new BBacnetEnumTrendLogRemoteExt() :
            new BBacnetEnumTrendLogExt();
        }

      case BacnetConst.ASN_NULL:
      case BacnetConst.ASN_OCTET_STRING:
      case BacnetConst.ASN_CHARACTER_STRING:
      case BacnetConst.ASN_DATE:
      case BacnetConst.ASN_TIME:
      case BacnetConst.ASN_OBJECT_IDENTIFIER:
      case BacnetConst.ASN_CONSTRUCTED_DATA:
      case BacnetConst.ASN_BACNET_ARRAY:
      case BacnetConst.ASN_BACNET_LIST:
      case BacnetConst.ASN_ANY:
      case BacnetConst.ASN_CHOICE:
      case BacnetConst.ASN_UNKNOWN_PROPRIETARY:
        return isRemote ?
          new BBacnetStringTrendLogRemoteExt() :
          new BBacnetStringTrendLogExt();

      default:
        throw new BacnetException("BACnet property type " + BBacnetPropertyIdentifier.tag(objectType)
          + " is not supported when making a trend log extension for " +
          (isRemote ? "remote" : "local") + " device");
    }
  }

  public static boolean isGenericTrendLogExtension(BIBacnetTrendLogExt ext)
  {
    return ext instanceof BBacnetNumericTrendLogExt ||
           ext instanceof BBacnetStringTrendLogExt ||
           ext instanceof BBacnetEnumTrendLogExt ||
           ext instanceof BBacnetBooleanTrendLogExt ||
           ext instanceof BBacnetTrendLogRemoteExt;
  }

  /**
   * This method calculates the next oid that can be used by a dynamic object for the particular object type.
   *
   * @param objectType The type of object that needs to be created
   * @return BBacnetObjectIdentifier the next oid to be used for dynamic object creation
   */
  public static BBacnetObjectIdentifier nextObjectIdentifier(int objectType)
  {
    BBacnetExportTable et = exportTable();
    BBacnetObjectIdentifier oid;
    for (int i = 0; i < BBacnetObjectIdentifier.MAX_INSTANCE_NUMBER; i++)
    {
      oid = BBacnetObjectIdentifier.make(objectType, i);
      if (et.byObjectId(oid) == null)
      {
        return oid;
      }
    }

    return null;
  }

  /**
   * Simple utility to ge the exportTable of bacnet network
   *
   * @return The exportTable
   */
  public static BBacnetExportTable exportTable()
  {
    return ((BBacnetExportTable)BBacnetNetwork.localDevice().getExportTable());
  }

  public static ChangeListError makeAddListElementError(BBacnetErrorClass errorClass, BBacnetErrorCode errorCode)
  {
    return new NChangeListError(BacnetConfirmedServiceChoice.ADD_LIST_ELEMENT, new NErrorType(errorClass, errorCode), 0);
  }

  public static ChangeListError makeRemoveListElementError(BBacnetErrorClass errorClass, BBacnetErrorCode errorCode)
  {
    return new NChangeListError(BacnetConfirmedServiceChoice.REMOVE_LIST_ELEMENT, new NErrorType(errorClass, errorCode), 0);
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private static final Logger logger = Logger.getLogger("bacnet.export.object.util");
  private static final BControlPoint[] EMPTY_POINT_ARRAY = new BControlPoint[0];

  private static final String DISCRETE_TOTALIZER_EXT = "DiscreteTotalizerExtension";
}
