/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define([], function () {
  'use strict';

  var LAYOUT_ABS = 0;
  var LAYOUT_PERCENT = 1;

  /**
   * @private
   * @exports mobile/px/gx/gxStyles
   */
  var exports = {};

  /**
   * Apply CSS styles from a Simple such as a Color or Font.
   *
   * @param {baja.Simple} simple
   * @param {...*} args
   */
  exports.applyCssFromSimple = function (simple) {
    for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      args[_key - 1] = arguments[_key];
    }
    switch (String(simple.getType())) {
      case 'gx:Brush':
        return exports.applyBrush.apply(exports, [simple].concat(args));
      case 'gx:Color':
        return exports.applyColor.apply(exports, [simple].concat(args));
      case 'gx:Font':
        return exports.applyFont.apply(exports, [simple].concat(args));
      case 'gx:Insets':
        return exports.applyInsets.apply(exports, [simple].concat(args));
      case 'bajaui:Layout':
        return exports.applyLayout.apply(exports, [simple].concat(args));
      default:
        return simple.update.apply(simple, args);
    }
  };

  /**
   * @param {baja.Simple} brush a `gx:Brush`
   * @param {object} obj object used to update using the brush's Paint instance
   */
  exports.applyBrush = function (brush, obj) {
    brush.$paint.update(obj);
  };

  /**
   * @param {baja.Simple} color a `gx:Color`
   * @param {object} obj
   * @param {JQuery} obj.dom
   * @param {String} obj.colorSelector what CSS property to update, e.g. `background-color`
   * @param {String} [obj.prefix] prefix to apply to the color selector
   */
  exports.applyColor = function (color, obj) {
    var css = color.getCss();
    var prefix = css && obj.prefix || "";
    obj.dom.css(obj.colorSelector, prefix + css);
  };

  /**
   * @param {baja.Simple} font a `gx:Font`
   * @param {JQuery} dom
   */
  exports.applyFont = function (font, dom) {
    dom.css({
      "font": font.getCssString(),
      "text-decoration": font.getUnderline() ? "underline" : "none"
    });
  };

  /**
   * @param {baja.Simple} insets a `gx:Insets`
   * @param {JQuery} dom
   * @param {Number} [excess=0] the excess to add onto to any padding in all directions
   * @param {String} [insetsType='padding'] what to update in the CSS
   */
  exports.applyInsets = function (insets, dom, excess, insetsType) {
    excess = excess || 0;
    insetsType = insetsType || "padding";
    var obj = {};
    obj[insetsType + "-top"] = insets.getTop() + excess;
    obj[insetsType + "-right"] = insets.getRight() + excess;
    obj[insetsType + "-bottom"] = insets.getBottom() + excess;
    obj[insetsType + "-left"] = insets.getLeft() + excess;
    dom.css(obj);
  };

  /**
   * @param {baja.Simple} layout a `bajaui:Layout`
   * @param {JQuery} dom
   */
  exports.applyLayout = function (layout, dom) {
    var wUnit = layout.getWidthUnit();
    var w = layout.getWidth();
    var hUnit = layout.getHeightUnit();
    var h = layout.getHeight();
    var xUnit = layout.getXUnit();
    var x = layout.getX();
    var yUnit = layout.getYUnit();
    var y = layout.getY();
    var width = "0",
      height = "0";
    if (wUnit === LAYOUT_PERCENT) {
      width = w + "%";
    } else if (wUnit === LAYOUT_ABS) {
      width = w;
    }
    if (hUnit === LAYOUT_PERCENT) {
      height = h + "%";
    } else if (hUnit === LAYOUT_ABS) {
      height = h;
    }

    // Absolutely position the child according to the layout information
    dom.css({
      "position": "absolute",
      "top": yUnit === LAYOUT_PERCENT ? y + "%" : y,
      "left": xUnit === LAYOUT_PERCENT ? x + "%" : x,
      "width": width,
      "height": height
    });
  };
  return exports;
});
