/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview CanvasPane class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'jquery', 'jquerymobile', 'mobile/px/util.px', 'mobile/px/widgets/Widget', 'mobile/px/gx/Layout'], function defineCanvasPane(baja, $, jqm, pxUtil, Widget, Layout) {
  "use strict";

  var superLoadChildren = Widget.prototype.loadChildren,
    callSuper = baja.callSuper,
    addScaling = pxUtil.addScaling,
    getScaledLayoutCSS = pxUtil.getScaledLayoutCSS;

  /**
   * Applies the appropriate dimension layout for the canvas pane's scale mode.
   * Works in both absolute and fluid modes, but as of right now, only
   * 'fitRatio' and 'none' are appropriate values for fluid.
   * 
   * @private
   * @inner
   * @ignore
   * @param {CanvasPane} canvas
   * @param {Boolean} [looping] sometimes scroll bars appear in a fluid layout
   * which necessitates a recalc - safety measure to prevent infinite loop 
   */
  function scaleCanvasPane(canvas, looping) {
    var dom = canvas.getDomElement(),
      isAbsolute = canvas.isAbsolutelyPositioned(),
      inner = canvas.$inner,
      parent = dom.parent(),
      domWidth = dom.width(),
      parentWidth = parent.width(),
      viewSize = canvas.getViewSize(),
      layout = canvas.getLayout(),
      outerWidth,
      outerHeight,
      viewWidth = viewSize.getWidth(),
      viewHeight = viewSize.getHeight(),
      viewAspectRatio,
      scale = canvas.getScale().getTag(),
      css;

    // If the CanvasPane is absolutely positioned then use the layout
    if (isAbsolute) {
      if (layout === Layout.FILL) {
        outerWidth = viewWidth;
        outerHeight = viewHeight;
      } else {
        outerWidth = layout.getWidth();
        outerHeight = layout.getHeight();
      }
    } else {
      // otherwise use the DOM for width
      outerWidth = domWidth;
      if (scale === 'fitRatio') {
        viewAspectRatio = viewWidth / viewHeight;
        outerHeight = outerWidth / viewAspectRatio;
      } else {
        // ONLY fitRatio and none are supported in fluid layout.
        scale = 'none';
        outerHeight = viewHeight;
      }
    }
    css = getScaledLayoutCSS(outerWidth, outerHeight, viewWidth,
    //innerWidth 
    viewHeight,
    //innerHeight
    canvas.getHalign().getTag(),
    //halign
    canvas.getValign().getTag(),
    //valign
    scale);
    addScaling(css, css.width / viewWidth, css.height / viewHeight);
    inner.css(css);

    /*
     * Sometimes, setting the canvas pane dimensions in a fluid page adds a 
     * scroll bar. this can change our domWidth, but innerCanvasPane
     * must then be recalculated.
     */
    if (!isAbsolute && parentWidth !== parent.width()) {
      if (!looping) {
        //only try once
        scaleCanvasPane(canvas, true);
      }
    }
  }

  /**
   * If the canvas pane is in a fluid layout, its inner div is first set to
   * width/height of 0 (so that its containing grid can re-calc column widths).
   * This is needed because scaling in a fluid layout depends on the canvas
   * pane's parent's dimensions. Then asynchronously performs scaling.
   * 
   * @private
   * @inner
   * @ignore
   * @param {CanvasPane} canvas
   */
  function performCanvasPaneRescale(canvas) {
    if (canvas.isAbsolutelyPositioned()) {
      scaleCanvasPane(canvas);
    } else {
      var inner = canvas.$inner;
      inner.css({
        width: 0,
        height: 0
      });

      //run async so we get a browser reflow. it's a little clunky but should
      //only happen upon window resize
      baja.runAsync(function () {
        scaleCanvasPane(canvas);
      });
    }
  }

  /**
   * @class CanvasPane.
   * 
   * The mobile pane Widget that represents Niagara's 'bajaui:CanvasPane' Type.
   * 
   * This Widget will absolutely position Widgets in a Px page.
   * 
   * @name CanvasPane
   * @extends Widget
   */
  function CanvasPane() {
    callSuper(CanvasPane, this, arguments);
  }
  baja.subclass(CanvasPane, Widget);

  /**
   * Append the Widget's DOM content onto the specified DOM element.
   *
   * @param {Object} dom DOM element to append content too.
   */
  CanvasPane.prototype.load = function (dom) {
    var that = this,
      autoScale,
      rootDom,
      windowWidth;
    that.$dom = $("<div class='canvasPane'><div class='innerCanvasPane'></div></div>");
    that.$inner = this.$dom.find(".innerCanvasPane");
    that.$dom.appendTo(dom);
    autoScale = baja.throttle(function autoScale() {
      var w = $(window).width();
      if (w !== windowWidth) {
        performCanvasPaneRescale(that);
      }
      windowWidth = w;
    }, 500);

    // Only do this if the Widget is not absolutely positioned. We do this so the
    // CanvasPane will automatically shrink and contract to the window.    
    if (!that.isAbsolutelyPositioned()) {
      rootDom = that.getRootContainer().getDomElement();
      rootDom.one("pxPageLoaded", autoScale);

      // When the page is started, register the auto scale function on window resize
      rootDom.on("pxPageStarted", function () {
        $(window).resize(autoScale);
      });

      // When the page is stopped remove this resize handler from the window
      rootDom.on("pxPageStopped", function () {
        $(window).off("resize", autoScale);
      });
    }
  };

  /**
   * Update the DOM associated with this Widget.
   *
   * @param {Function} hasUpdate called to test whether a 
   *                               given Property can update the DOM.
   * @param {Boolean} [firstUpdate] set to true on Widgets first update
   *                                when the page is first loaded.
   */
  CanvasPane.prototype.doUpdate = function (hasUpdate, firstUpdate) {
    var that = this,
      dom = that.$dom;

    // Background color
    if (hasUpdate("background")) {
      that.applyCssFromSlot("background", {
        dom: dom,
        colorSelector: "background",
        imageSelector: "background"
      });
    }

    // View Size
    if (hasUpdate("viewSize")) {
      performCanvasPaneRescale(that);
      // If there is a parent then also update the layout
      if (!firstUpdate && that.getParent() && that.getDomElement()) {
        // If the parent Widget supports laying out children then call it
        that.getParent().layoutChild(that);
      }
    } else if (hasUpdate("halign") || hasUpdate("valign") || hasUpdate("scale")) {
      performCanvasPaneRescale(that);
    }
  };

  /**
   * Update the positioning of the child Widget
   *
   * @param {Widget} childWidget the Child Widget to be positioned.
   */
  CanvasPane.prototype.layoutChild = function (childWidget) {
    childWidget.applyCssFromSlot('layout', childWidget.getDomElement());

    // TODO: Z-order, change of Z-order etc.
  };

  /**
   * Return the preferred Width of the CanvasPane.
   *
   * @returns {Number} the width in pixels.
   */
  CanvasPane.prototype.getPreferredWidth = function () {
    return this.getViewSize().getWidth();
  };

  /**
   * Return the preferred Height of the CanvasPane.
   *
   * @returns {Number} the height in pixels.
   */
  CanvasPane.prototype.getPreferredHeight = function () {
    return this.getViewSize().getHeight();
  };

  /**
   * Load the child Widgets.
   * 
   * {@link baja.Widget#load} must be called before this method is invoked.
   */
  CanvasPane.prototype.loadChildren = function () {
    superLoadChildren.call(this, this.$inner);
  };

  /**
   * A CanvasPane should absolutely position itself if it is the child of
   * another CanvasPane (as usual) or if it itself is the root child of a
   * RootContainer.
   * @returns {boolean}
   */
  CanvasPane.prototype.isAbsolutelyPositioned = function () {
    if (baja.hasType(this.getParent(), 'bajaui:RootContainer')) {
      return true;
    }
    return Widget.prototype.isAbsolutelyPositioned.apply(this, arguments);
  };
  return CanvasPane;
});
