/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/* eslint-env browser */

define(['baja!', 'underscore', 'Promise'], function (baja, _, Promise) {
  'use strict';

  var each = _.each;

  /**
   * API Status: **Private**
   * @exports nmodule/bajaui/rc/util/pxUtils
   */
  var exports = {};
  exports.$blinkingElements = [];
  exports.$blinkInterval = null;
  exports.$blinkCounter = 0;
  exports.$elementVisibility = 'hidden';
  var blinkFunction = function blinkFunction() {
    // visible for 600 ms, hidden for 200 ms
    exports.$blinkCounter = ++exports.$blinkCounter % 8;
    exports.$elementVisibility = exports.$blinkCounter >= 2 ? '' : 'hidden';
    for (var i = 0; i < exports.$blinkingElements.length; i++) {
      if (document.body.contains(exports.$blinkingElements[i])) {
        exports.$blinkingElements[i].style.visibility = exports.$elementVisibility;
      } else {
        // remove element from array if no longer found in DOM
        exports.$blinkingElements.splice(i, 1);
        if (exports.$blinkingElements.length === 0) {
          clearInterval(exports.$blinkInterval);
          exports.$blinkInterval = null;
          exports.$blinkCounter = 0;
          exports.$elementVisibility = 'hidden';
          return;
        }
        i--;
      }
    }
  };

  /**
   * @param {module:nmodule/bajaui/rc/model/UxModel} model
   * @param {object} targetMap a mapping of Px Property names to their widget
   * paths (arrays of slot names)
   * @param {object} valueMap a mapping of Px Property names to their overridden
   * values
   */
  exports.applyPxProperties = function (model, targetMap, valueMap) {
    if (!targetMap || !valueMap) {
      return;
    }
    each(targetMap, function (_ref, name) {
      var targets = _ref.targets;
      each(targets, function (path) {
        var prop = path.pop();
        var kid = model.get(path);
        if (kid) {
          var value = valueMap[name];
          if (typeof value !== 'undefined') {
            if (baja.hasType(kid, 'bajaui:Binding')) {
              kid.set({
                slot: prop,
                value: value
              });
            } else {
              kid.getProperties()[prop] = value;
            }
          }
        }
      });
    });
  };

  /**
   * Make DOM elements start or stop blinking. By maintaining a central list of the blinking items,
   * this will prevent the blinking items from blinking at different times.
   *
   * @see {module:nmodule/hx/com/tridium/hx/px/hxPx.toggleBlink}
   */
  exports.updateBlink = function (elements, blink) {
    if (!blink && exports.$blinkingElements.length === 0) {
      return;
    }
    for (var i = 0; i < elements.length; i++) {
      if (!blink && elements[i]) {
        var index = exports.$blinkingElements.indexOf(elements[i]);
        if (index > -1) {
          exports.$blinkingElements.splice(index, 1);
          if (exports.$blinkingElements.length === 0) {
            clearInterval(exports.$blinkInterval);
            exports.$blinkInterval = null;
            exports.$blinkCounter = 0;
            exports.$elementVisibility = 'hidden';
          }
          elements[i].style.visibility = '';
        }
      } else {
        if (!elements[i] || exports.$blinkingElements.indexOf(elements[i]) > -1) {
          continue;
        }

        // sync newly blinking element up with the rest of the blinking elements
        elements[i].style.visibility = exports.$elementVisibility;
        exports.$blinkingElements.push(elements[i]);
        if (!exports.$blinkInterval && exports.$blinkingElements.length === 1) {
          exports.$blinkInterval = setInterval(blinkFunction, 100);
        }
      }
    }
  };

  /**
   * @param {string|baja.Ord} ord
   * @returns {baja.Ord} an ORD with the view query removed, if present
   */
  exports.removeViewQuery = function (ord) {
    if (!ord) {
      return baja.Ord.DEFAULT;
    }
    var queries = new baja.OrdQueryList();
    baja.Ord.make(ord).parse().getCursor().each(function (value) {
      var scheme = value.getSchemeName();
      if (scheme !== 'view') {
        queries.add(value);
      }
    });
    return baja.Ord.make(queries);
  };

  /**
   * Recursively update all bajaux Properties, and Binding Properties, that are
   * type ORD, with the appropriate ORD variable substitutions.
   * @param {module:nmodule/bajaui/rc/model/UxModel} model
   * @param {baja.Facets|object} [variables]
   * @returns {Promise.<module:nmodule/bajaui/rc/model/UxModel>} to be resolved
   * with the updated model
   */
  exports.substituteOrds = function (model, variables) {
    var props = model.getProperties();
    Object.keys(props).forEach(function (name) {
      var value = props[name];
      if (isOrd(value)) {
        props[name] = value.substitute(variables);
      }
    });
    return Promise.all(model.getBindingList().getBindings().map(function (binding) {
      return substituteOrds(binding, variables);
    })).then(function () {
      return Promise.all(model.getKids().map(function (kid) {
        return exports.substituteOrds(kid, variables);
      }));
    }).then(function () {
      return model;
    });
  };

  /**
   * Retrieve the scroll bar width for the current browser. If the `ux-PxWidget` CSS class is
   * available on an element in the document, then use that element as the location for the test
   * to try help ensure that any theming to the scrollbar is taken into account.
   * When a `ux-PxWidget` CSS class is is found on a DOM element, the result will be cached.
   * @return {number}
   */
  exports.getScrollBarWidth = function () {
    if (exports.$scrollBarWidth !== undefined) {
      return exports.$scrollBarWidth;
    }
    // Create the measurement node
    var scrollDiv = document.createElement("div");
    scrollDiv.className = "-t-scrollbar-measure";
    var cacheMeasurement = false;
    var measureParent = document.body;
    var pxWidget = document.getElementsByClassName('ux-PxWidget');
    if (pxWidget.length > 0) {
      measureParent = pxWidget[0];
      cacheMeasurement = true;
    }
    measureParent.appendChild(scrollDiv);
    var scrollbarWidth = scrollDiv.offsetWidth - scrollDiv.clientWidth;
    if (cacheMeasurement) {
      exports.$scrollBarWidth = scrollbarWidth;
    }
    measureParent.removeChild(scrollDiv);
    return scrollbarWidth;
  };

  /**
   * Retrieve info about whether a zoom offset needs to be utilized during ScaledLayout computations.
   * This is determined by finding the closest DOM element with the `-t-scaled-layout-scroll-container`
   * CSS className. If that container has an `auto` overflow, then the zoomOffset will be taken into account
   * for the appropriate dimension.
   *
   * @param {String} scaleTag
   * @param {JQuery} jq
   * @return {{applyZoomOffsetWidth: boolean, applyZoomOffsetHeight: boolean }}
   */
  exports.getApplyZoomOffsetInfo = function (scaleTag, jq) {
    var result = {
      applyZoomOffsetWidth: false,
      applyZoomOffsetHeight: false
    };
    if (!scaleTag.startsWith('zoom')) {
      return result;
    }
    var scrollPaneJq = jq.closest('.-t-scaled-layout-scroll-container');
    if (scrollPaneJq.length > 0) {
      if (scrollPaneJq.css('overflow-y') === 'auto') {
        result.applyZoomOffsetWidth = true;
      }
      if (scrollPaneJq.css('overflow-x') === 'auto') {
        result.applyZoomOffsetHeight = true;
      }
    }
    return result;
  };
  function substituteOrds(comp, variables) {
    var props = comp.getSlots().properties().toArray();
    return Promise.all(props.map(function (slot) {
      var val = comp.get(slot);
      return isOrd(val) && comp.set({
        slot: slot,
        value: val.substitute(variables)
      });
    }));
  }
  function isOrd(v) {
    return baja.hasType(v, 'baja:Ord');
  }
  return exports;
});
