/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/baja/AlarmTransitionBits
 */
define(['baja!', 'underscore'], function (baja, _) {
  'use strict';

  var objUtil = require('bajaScript/baja/obj/objUtil');
  var Simple = baja.Simple,
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode,
    STR_REGEX = /^[0-9A-F]+$/i;

  /**
   * BajaScript representation of an `alarm:AlarmTransitionBits` value.
   *
   * @class
   * @extends baja.Simple
   * @alias module:nmodule/alarm/rc/baja/AlarmTransitionBits
   */
  var AlarmTransitionBits = function AlarmTransitionBits() {
    Simple.apply(this, arguments);
    this.$bits = 0;
  };
  AlarmTransitionBits.prototype = Object.create(Simple.prototype);
  AlarmTransitionBits.prototype.constructor = AlarmTransitionBits;

  /**
   * Create a new `AlarmTransitionBits` instance from the individual bits to
   * set.
   *
   * @returns {module:nmodule/alarm/rc/baja/AlarmTransitionBits}
   */
  AlarmTransitionBits.make = function () {
    var at = new AlarmTransitionBits();
    at.$bits = _.reduce(arguments, function (memo, bool, i) {
      return bool ? memo + (1 << i) : memo;
    }, 0);
    return at;
  };

  /**
   * @see .make
   */
  AlarmTransitionBits.prototype.make = function () {
    return AlarmTransitionBits.make.apply(AlarmTransitionBits, arguments);
  };

  /**
   * Decode an `AlarmTransitionBits` from a hex number string.
   *
   * @function
   * @param {String} str hex number string
   */
  AlarmTransitionBits.prototype.decodeFromString = cacheDecode(function (str) {
    if (!STR_REGEX.exec(str)) {
      throw new Error('invalid AlarmTransitionBits encoding: ' + str);
    }
    var bits = new AlarmTransitionBits();
    bits.$bits = parseInt(str, 16);
    return bits;
  });

  /**
   * Encode this `AlarmTransitionBits` to a hex number string.
   *
   * @function
   * @returns {String} hex number string
   */
  AlarmTransitionBits.prototype.encodeToString = cacheEncode(function () {
    return this.$bits.toString(16);
  });

  /**
   * Return true if this `AlarmTransitionBits` is to alert.
   * @returns {boolean}
   */
  AlarmTransitionBits.prototype.isToAlert = function () {
    return !!(this.$bits & AlarmTransitionBits.TO_ALERT);
  };

  /**
   * Return true if this `AlarmTransitionBits` is to fault.
   * @returns {boolean}
   */
  AlarmTransitionBits.prototype.isToFault = function () {
    return !!(this.$bits & AlarmTransitionBits.TO_FAULT);
  };

  /**
   * Return true if this `AlarmTransitionBits` is to normal.
   * @returns {boolean}
   */
  AlarmTransitionBits.prototype.isToNormal = function () {
    return !!(this.$bits & AlarmTransitionBits.TO_NORMAL);
  };

  /**
   * Return true if this `AlarmTransitionBits` is to offnormal.
   * @returns {boolean}
   */
  AlarmTransitionBits.prototype.isToOffNormal = function () {
    return !!(this.$bits & AlarmTransitionBits.TO_OFFNORMAL);
  };

  /**
   * Return a string representation of this `AlarmTransitionBits`.
   *
   * @param {Object} [cx] a context object. The `cx` can indicate what bits to hide
   * from the output. Set values for `AlarmTransitionBits.SHOW_ALERT`, etc. to
   * `false` to hide those bits.
   * @returns {string}
   */
  AlarmTransitionBits.prototype.toString = function (cx) {
    var bits = this.$bits,
      displayKeys = ['alert', 'toOffNormal', 'toFault', 'toNormal'],
      showKeys = [AlarmTransitionBits.SHOW_ALERT, AlarmTransitionBits.SHOW_OFF_NORMAL, AlarmTransitionBits.SHOW_FAULT, AlarmTransitionBits.SHOW_NORMAL];
    return '{' + _.filter(displayKeys, function (key, i) {
      return bits & 1 << i && (!cx || cx[showKeys[i]] !== false);
    }).join(' ') + '}';
  };

  /**
   * Return a baja.EnumSet representation of this 'AlarmTransitionBits' where the EnumSet
   * ordinals represent the currently set bits
   *
   * @returns {baja.EnumSet} EnumSet representation of alarm transition bits
   */
  AlarmTransitionBits.prototype.toEnumSet = function () {
    var bits = this.$bits,
      ordinals = [AlarmTransitionBits.TO_OFFNORMAL, AlarmTransitionBits.TO_FAULT, AlarmTransitionBits.TO_NORMAL, AlarmTransitionBits.TO_ALERT],
      tags = ['toOffNormal', 'toFault', 'toNormal', 'toAlert'],
      range = baja.EnumRange.make({
        ordinals: ordinals,
        tags: tags
      }),
      activeBits = _.filter(ordinals, function (ordinal) {
        return bits & ordinal;
      });
    return baja.EnumSet.make({
      ordinals: activeBits,
      range: range
    });
  };

  /**
   * Facet string indicating that the off normal bit should be shown.
   * @type {string}
   */
  AlarmTransitionBits.SHOW_OFF_NORMAL = 'showOffNormal';

  /**
   * Facet string indicating that the fault bit should be shown.
   * @type {string}
   */
  AlarmTransitionBits.SHOW_FAULT = 'showFault';

  /**
   * Facet string indicating that the normal bit should be shown.
   * @type {string}
   */
  AlarmTransitionBits.SHOW_NORMAL = 'showNormal';

  /**
   * Facet string indicating that the alert bit should be shown.
   * @type {string}
   */
  AlarmTransitionBits.SHOW_ALERT = 'showAlert';

  /**
   * Indicates to offnormal.
   * @type {number}
   */
  AlarmTransitionBits.TO_OFFNORMAL = 0x01;

  /**
   * Indicates to fault.
   * @type {number}
   */
  AlarmTransitionBits.TO_FAULT = 0x02;

  /**
   * Indicates to normal.
   * @type {number}
   */
  AlarmTransitionBits.TO_NORMAL = 0x04;

  /**
   * Indicates to alert.
   * @type {number}
   */
  AlarmTransitionBits.TO_ALERT = 0x08;

  /**
   * Indicates to offnormal.
   * @type {module:nmodule/alarm/rc/baja/AlarmTransitionBits}
   */
  AlarmTransitionBits.toOffnormal = AlarmTransitionBits.make(true, false, false, false);

  /**
   * Indicates to fault.
   * @type {module:nmodule/alarm/rc/baja/AlarmTransitionBits}
   */
  AlarmTransitionBits.toFault = AlarmTransitionBits.make(false, true, false, false);

  /**
   * Indicates to normal.
   * @type {module:nmodule/alarm/rc/baja/AlarmTransitionBits}
   */
  AlarmTransitionBits.toNormal = AlarmTransitionBits.make(false, false, true, false);

  /**
   * Indicates to alert.
   * @type {module:nmodule/alarm/rc/baja/AlarmTransitionBits}
   */
  AlarmTransitionBits.toAlert = AlarmTransitionBits.make(false, false, false, true);

  /**
   * `DEFAULT` has all bits set.
   * @type {module:nmodule/alarm/rc/baja/AlarmTransitionBits}
   */
  AlarmTransitionBits.DEFAULT = AlarmTransitionBits.ALL = AlarmTransitionBits.make(true, true, true, true);

  /**
   * `EMPTY` has no bits set.
   * @type {module:nmodule/alarm/rc/baja/AlarmTransitionBits}
   */
  AlarmTransitionBits.EMPTY = AlarmTransitionBits.make(false, false, false, false);
  baja.registerType('alarm:AlarmTransitionBits', function () {
    return AlarmTransitionBits;
  });
  return AlarmTransitionBits;
});
