/*jshint browser: true */ /* eslint-env browser */

/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/AlarmConsole
 */
define(['lex!alarm', 'log!nmodule.alarm.rc.console.AlarmConsole', 'baja!', 'jquery', 'Promise', 'underscore', 'bajaux/commands/CommandGroup', 'bajaux/events', 'bajaux/mixin/responsiveMixIn', 'bajaux/util/CommandButtonGroup', 'nmodule/alarm/rc/Alarm', 'nmodule/alarm/rc/console/AlarmConsoleViewModel', 'nmodule/alarm/rc/console/commands/AcknowledgeCommand', 'nmodule/alarm/rc/console/commands/DetailsCommand', 'nmodule/alarm/rc/console/commands/FilterCommand', 'nmodule/alarm/rc/console/commands/ForceClearCommand', 'nmodule/alarm/rc/console/commands/HyperlinkCommand', 'nmodule/alarm/rc/console/commands/NotesCommand', 'nmodule/alarm/rc/console/commands/OptionsCommand', 'nmodule/alarm/rc/console/commands/ShowAlarmsCommand', 'nmodule/alarm/rc/console/commands/SilenceCommand', 'nmodule/alarm/rc/console/commands/SoundOffCommand', 'nmodule/alarm/rc/console/commands/ContinuousCommand', 'nmodule/alarm/rc/console/commands/SelectAllCommand', 'nmodule/alarm/rc/console/table/AlarmTableContextMenu', 'nmodule/alarm/rc/util/alarmUtils', 'nmodule/gx/rc/baja/Color', 'nmodule/gx/rc/util/colorUtils', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/util/decorate', 'nmodule/webEditors/rc/util/ListSelection', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport', 'nmodule/webEditors/rc/wb/table/pagination/PaginationModel', 'nmodule/webEditors/rc/wb/table/pagination/PaginationWidget', 'nmodule/webEditors/rc/wb/table/Table', 'hbs!nmodule/alarm/rc/console/templates/AlarmConsole', 'css!nmodule/alarm/rc/console/templates/alarmConsole'], function (lexs, log, baja, $, Promise, _, CommandGroup, events, responsiveMixIn, CommandButtonGroup, Alarm, AlarmConsoleViewModel, AcknowledgeCommand, DetailsCommand, FilterCommand, ForceClearCommand, HyperlinkCommand, NotesCommand, OptionsCommand, ShowAlarmsCommand, SilenceCommand, SoundOffCommand, ContinuousCommand, SelectAllCommand, AlarmTableContextMenu, alarmUtils, Color, colorUtils, BaseWidget, fe, decorate, ListSelection, contextMenuSupport, PaginationModel, PaginationWidget, Table, template) {
  'use strict';

  var hasOperatorWrite = alarmUtils.hasOperatorWrite,
    lex = lexs[0],
    logSevere = log.severe.bind(log),
    MULTI_SOURCE_COUNTS_LEX_KEY = "alarm.console.counts.multiSource",
    SINGLE_SOURCE_COUNTS_LEX_KEY = "alarm.console.counts.singleSource",
    PAGINATION_PREFIX_KEY = "alarm.console.pagination.prefix",
    PAGINATION_SUFFIX_KEY = "alarm.console.pagination.suffix",
    TIME_RANGE_SELECTOR = ".timeRange",
    COMMAND_BAR_SELECTOR = '.commands',
    TABLE_CONTAINER_SELECTOR = '.alarmTable',
    SHOW_HIDE_MENU_SELECTOR = '.showHideMenu',
    PAGINATION_CONTAINER_SELECTOR = '.paginationContainer',
    PAGINATION_WIDGET_SELECTOR = '.paginationWidget',
    PAGE_COUNT_SELECTOR = '.pageCount',
    COUNT_SUMMARY_SELECTOR = '.countSummary',
    COUNTS_CHANGED_EVENT = 'countsChanged',
    RECIPIENT_UPDATE_EVENT = 'recipientUpdate';
  function initTimeRangeEditor(view) {
    var viewModel = view.$viewModel;
    return baja.TimeZoneDatabase.get().then(function (database) {
      var timezoneId = Intl.DateTimeFormat().resolvedOptions().timeZone,
        timezone = timezoneId && database.getTimeZone(timezoneId);
      return fe.buildFor({
        dom: $(TIME_RANGE_SELECTOR, view.jq()),
        formFactor: 'mini',
        value: viewModel.$getTimeRange(),
        properties: {
          TimeZone: timezone && timezone.encodeToString()
        }
      });
    }).then(function (editor) {
      view.$timeRangeEditor = editor;
      editor.jq().on(events.MODIFY_EVENT, function () {
        editor.read().then(function (timeRange) {
          viewModel.timeRangeChanged(timeRange);
        })["catch"](logSevere);
      });
    });
  }
  function addToCommandGroup(view, cmd) {
    view.getCommandGroup().add(cmd);
    return cmd;
  }
  function fireSelectionChangedOnCommands(view) {
    view.getCommandGroup().flatten().forEach(function (cmd) {
      if (typeof cmd.selectionChanged === 'function') {
        cmd.selectionChanged();
      }
    });
  }
  function initCommands(view) {
    var cmdGroup = view.getCommandGroup();
    view.$ackCmd = addToCommandGroup(view, new AcknowledgeCommand(view));
    view.$ackRecentCmd = addToCommandGroup(view, new AcknowledgeCommand(view, {
      mostRecent: true
    }));
    view.$hyperlinkCmd = addToCommandGroup(view, new HyperlinkCommand(view));
    view.$notesCmd = addToCommandGroup(view, new NotesCommand(view));
    view.$forceClearCmd = addToCommandGroup(view, new ForceClearCommand(view));
    view.$detailsCmd = addToCommandGroup(view, new DetailsCommand(view));
    view.$silenceCmd = addToCommandGroup(view, new SilenceCommand(view));
    view.$filterCmd = addToCommandGroup(view, new FilterCommand(view));
    view.$soundOffCmd = addToCommandGroup(view, new SoundOffCommand(view));
    view.$continuousCmd = addToCommandGroup(view, new ContinuousCommand(view));
    view.$optionsCmd = addToCommandGroup(view, new OptionsCommand(view));
    view.$selectAllCmd = addToCommandGroup(view, new SelectAllCommand(view));
    view.$showAlarmsCmd = addToCommandGroup(view, new ShowAlarmsCommand(view));
    view.$soundOffCmd.on(events.command.SELECTION_EVENT, function () {
      var state = !view.$soundOffCmd.isSelected();
      if (state !== view.$continuousCmd.isEnabled()) {
        view.$continuousCmd.setEnabled(state);
      }
    });
    view.$soundOffCmd.update();
    view.$continuousCmd.update();
    return decorate(view, {
      typeSpec: 'alarm:AlarmUxConsole'
    }).then(function () {
      fireSelectionChangedOnCommands(view);
      var commandBarGroup = new CommandGroup({
        commands: cmdGroup.flatten().filter(function (cmd) {
          return _.result(cmd, 'isShownInCommandBar');
        })
      });
      return fe.buildFor({
        dom: $(COMMAND_BAR_SELECTOR, view.jq()),
        value: commandBarGroup,
        type: CommandButtonGroup,
        formFactor: 'max'
      });
    });
  }
  function getRowIndex(dom) {
    return $(dom).closest('.ux-table-row').index();
  }
  function initAlarmTable(view) {
    var viewModel = view.$viewModel,
      dom = view.jq(),
      summaryDom = $(COUNT_SUMMARY_SELECTOR, dom),
      tableDom = $(TABLE_CONTAINER_SELECTOR, dom),
      table = view.$table = new Table({
        formFactor: 'max',
        selection: new ListSelection(),
        properties: {
          fixedHeaders: true
        }
      }),
      selection = table.$getSelection(),
      paginationContainerDom = $(PAGINATION_CONTAINER_SELECTOR, dom).hide(),
      paginationDom = $(PAGINATION_WIDGET_SELECTOR, dom),
      pageCountDom = $(PAGE_COUNT_SELECTOR, dom),
      paginationModel = view.$paginationModel = new PaginationModel({
        sortingAllowed: true
      });
    function finishBuildingRow(row, dom) {
      var record = row.getSubject();
      var key;
      var color;
      var singleSrc = view.isSingleSourceView();
      var options = viewModel.$getAlarmConsoleOptions() || {};
      var viewOption = singleSrc ? options.singleSourceView : options.multiSourceView;

      // Add row highlight from an update.
      if (row.data("fromUpdate")) {
        //TODO: viewModel already knows if it's single source or not.
        // viewModel.getAlarmConsoleOptions() should just return me my options
        // without the isSingleSourceView rigmarole

        if (!(viewOption && viewOption.disableRowHighlight)) {
          if (record._ackState === "acked") {
            key = "acked";
          } else if (record._sourceState === "normal") {
            key = "normal";
          } else {
            key = "alarm";
          }
          color = lex.getSafe("alarm.state.color." + key);

          // Do the row animation.
          dom.css({
            "transition": "none",
            "box-shadow": "0px 0px 3px 1px " + color + " inset"
          });
          setTimeout(function () {
            dom.css({
              "transition": "box-shadow 2s",
              "box-shadow": "0px 0px 0px 0px " + color + " inset"
            });
          }, 20);
          setTimeout(function () {
            row.data("fromUpdate", false);
          }, 2000);
        }
      }

      //set the rows background and color based on priority
      var multiOptions = options.multiSourceView;
      if (multiOptions) {
        var lowColor = Color.make(multiOptions.lowPriorityColor);
        var midColor = Color.make(multiOptions.midPriorityColor);
        var highColor = Color.make(multiOptions.highPriorityColor);
        var nullColor = Color.make(null);

        // If all the colors are null then no need to set the row colors
        if (lowColor !== nullColor || midColor !== nullColor || highColor !== nullColor) {
          var backgroundColor = view.$computeBackgroundColor(record.priority, lowColor, midColor, highColor);
          var foregroundColor = Color.make("black");

          // if the alpha gets too low the getContrastingColor will return a white when black is actual needed
          if (backgroundColor.getAlpha() >= 50) {
            foregroundColor = Color.make(colorUtils.getContrastingColor(backgroundColor.getRgba()));
          }
          dom.css({
            "backgroundColor": backgroundColor.toCssString(),
            "color": foregroundColor.toCssString()
          });
        }
      }
      return Promise.resolve(dom);
    }

    //event handlers
    table.$sortByColumnDisplay = _.partial(sortByColumn, view);
    view.$armTableDomHandlers(tableDom, selection);
    selection.on('changed', function () {
      fireSelectionChangedOnCommands(view);

      // When the list selection changes, make sure all of the check boxes are
      // toggled correctly. The toggle on the checkboxes only work if we make
      // this an asynchronous call.
      setTimeout(function () {
        tableDom.find(".js-col-select > input").each(function () {
          var input = $(this),
            index = getRowIndex(input);
          input.prop("checked", selection.isSelected(index));
        });
      }, 0);
    });
    paginationModel.on('changed', function (key, value) {
      if (key === 'currentPage') {
        viewModel.pageIndexChanged(value)["catch"](logSevere);
      }
    });
    viewModel.on(COUNTS_CHANGED_EVENT, function (params) {
      if (params.noOfPages > 1) {
        paginationModel.setRowCount(params.alarmCount);
        pageCountDom.text(params.noOfPages);
        paginationContainerDom.show();
      } else {
        paginationContainerDom.hide();
      }

      //update source / alarm count summary

      function setSummaryText(text) {
        var parts = _.map(text.split(' / '), function (text) {
          return '<span class="summaryPart">' + text + '</span>';
        });
        summaryDom.html(parts.join('<span> / </span>'));
      }
      if (view.isSingleSourceView()) {
        setSummaryText(lex.getSafe(SINGLE_SOURCE_COUNTS_LEX_KEY, params.sourceName, params.alarmCount));
      } else {
        setSummaryText(lex.getSafe(MULTI_SOURCE_COUNTS_LEX_KEY, params.sourceCount, params.alarmCount));
      }
    });
    return Promise.all([table.initialize(tableDom, {
      finishBuildingRow: finishBuildingRow
    }), fe.buildFor({
      value: paginationModel,
      dom: paginationDom,
      formFactor: 'mini',
      type: PaginationWidget
    })]);
  }
  function sortByColumn(view, column, desc) {
    var viewModel = view.$viewModel,
      singleSrc = view.isSingleSourceView(),
      support = viewModel.$getSupport(),
      name = column.getName(),
      sortByCol = {
        column: name,
        desc: desc
      };
    return Promise.all([support.setAlarmConsoleOption('sortByColumn', sortByCol, singleSrc), singleSrc && viewModel.sortColumnChanged(name, desc)]).then(function () {
      if (!singleSrc) {
        sortTable(view, column, desc);
      }
    });
  }
  function armRowSorting(view) {
    var tableModel = view.$table.getModel(),
      sort = _.partial(sortTable, view);
    _.each(['rowsAdded', 'rowsChanged'], function (event) {
      tableModel.on(event, function () {
        if (!view.isSingleSourceView()) {
          var sortColumn = getSortColumn(view);
          sort(sortColumn.column, sortColumn.desc);
        }
      });
    });
  }
  function getSortColumn(view) {
    var table = view.$table,
      tableModel = table.getModel(),
      header = table.$getThead().children('th').filter(function () {
        var jq = $(this);
        return jq.hasClass('asc') || jq.hasClass('desc');
      });
    return header.length ? {
      column: header.data('column'),
      desc: header.hasClass('desc')
    } : {
      column: tableModel.getColumn('timestamp'),
      desc: true
    };
  }
  function sortTable(view, column, desc) {
    var tableModel = view.$table.value();
    tableModel.sort(function (r1, r2) {
      var sub1 = r1.getSubject(),
        sub2 = r2.getSubject(),
        t1 = sub1.timestamp,
        t2 = sub2.timestamp,
        v1 = column.getValueFor(r1),
        v2 = column.getValueFor(r2),
        s1,
        s2,
        result;
      if (typeof v1 === 'number' && typeof v2 === 'number') {
        //sort by descending timestamp for equal row values on non-timestamp column
        if (v1 === v2 && column.getName() !== 'timestamp') {
          return t2 - t1;
        }
        return desc ? v2 - v1 : v1 - v2;
      } else {
        s1 = String(v1);
        s2 = String(v2);
        result = s1.localeCompare(s2);
        if (result === 0) {
          //sort by descending timestamp for equal row values on non-timestamp column
          return t2 - t1;
        }
        return desc ? s2.localeCompare(s1) : result;
      }
    });
  }

  /**
   * The main view for managing live alarms in the Niagara system.
   *
   * @class
   * @alias module:nmodule/alarm/rc/console/AlarmConsole
   * @extends {module:nmodule/webEditors/rc/fe/BaseWidget}
   */
  var AlarmConsole = function AlarmConsole(moduleName, keyName, formFactor) {
    BaseWidget.call(this, {
      moduleName: moduleName,
      keyName: keyName,
      formFactor: formFactor
    });
    contextMenuSupport(this);
    responsiveMixIn(this, {
      'ux-alarm-console-mobile': {
        maxWidth: 800
      }
    });
  };
  AlarmConsole.prototype = Object.create(BaseWidget.prototype);
  AlarmConsole.prototype.constructor = AlarmConsole;
  AlarmConsole.prototype.$armTableDomHandlers = function (tableDom, selection) {
    var that = this,
      touchMoved = false,
      longPressTriggered = false,
      longPressTimerId = -1;
    function resetTouch() {
      longPressTriggered = false;
      touchMoved = false;
      clearTimeout(longPressTimerId);
      longPressTimerId = -1;
    }
    function showDetails(dom, e) {
      e.stopImmediatePropagation();
      selection.select(getRowIndex(dom));
      that.$detailsCmd.invoke()["catch"](logSevere);
    }

    // When a row is double clicked, show the alarm's details. This
    // will not work on a mobile device.
    tableDom.on('dblclick', 'tr', function (e) {
      if (!$(e.target).closest('.js-col-select').length) {
        showDetails(this, e);
      }
    });

    // If we have a touch move event then the user is assumed to be
    // scrolling the alarm list.
    tableDom.on('touchmove', function () {
      touchMoved = true;
    });
    tableDom.on('touchend', resetTouch);
    tableDom.on('touchstart', 'tr', function (e) {
      resetTouch();
      var context = this;
      longPressTimerId = setTimeout(function () {
        var target = $(e.target),
          originalEvent,
          touches,
          touch;

        // Only show the details if there's been no touch movement and
        // the row's check box hasn't been touched.
        if (!touchMoved && !target.closest('.js-col-select').length) {
          longPressTriggered = true;

          // If there's a long press then show the context menu.
          originalEvent = e.originalEvent;
          touches = originalEvent && originalEvent.touches;
          touch = touches && touches.length ? touches[0] : originalEvent;

          // Select any rows the user has pressed down before
          // triggering the context menu.
          selection.put(getRowIndex(context), true);
          selection.emit('changed');
          target.trigger($.Event('contextmenu', {
            which: 3,
            pageX: touch ? touch.pageX : 0,
            pageY: touch ? touch.pageY : 0,
            originalEvent: originalEvent
          }));
        }
      }, /*long press delay*/that.$longPressDelay || 1000);
    });

    // When a row is touched, show the alarm's details. If it's a long press
    // then show the context menu. This will only work on touch devices
    // where there is no double click available.
    tableDom.on('touchend', 'tr', function (e) {
      if (!touchMoved && !longPressTriggered && !$(e.target).closest('.js-col-select').length) {
        showDetails(this, e);
      }
      resetTouch();
    });

    // Toggle the row selection when the row's check box is clicked.
    tableDom.on('click', '.js-col-select', function (e) {
      e.stopImmediatePropagation();
      selection.toggle(getRowIndex(this));
    });

    // If the right mouse button is pressed on a row then
    // select the row in addition to what's already selected.
    tableDom.on('mousedown', 'tbody > tr', function (e) {
      if (e.which === 3) {
        selection.put(getRowIndex(this), true);
        selection.emit('changed');
      }
    });
  };

  /**
   * Initializes the time range selector, alarm table and commands
   *
   * @param {jQuery} dom
   * @param {Object} params
   * @returns {Promise} resolved when view has been initialized
   */
  AlarmConsole.prototype.doInitialize = function (dom, params) {
    var that = this;
    that.$viewModel = new AlarmConsoleViewModel(params);
    that.$viewModel.on(RECIPIENT_UPDATE_EVENT, function (summary, update) {
      if (update) {
        var icon = lex.get('alarm.console.commands.showAllNew.icon');
        icon = icon.replace(/^module:\/\//, "/module/");

        // At some point make bajaux Command only set an icon if it hasn't already changed.
        if (that.isSingleSourceView() && that.$showAlarmsCmd.getIcon() !== icon) {
          that.$showAlarmsCmd.setIcon(icon);
          that.$showAlarmsCmd.setDescriptionFormat(lex.get('alarm.console.commands.showAllNew.description'));
        }
      }
    });
    dom.addClass('AlarmConsole').html(template({
      prefix: lex.get(PAGINATION_PREFIX_KEY),
      suffix: lex.get(PAGINATION_SUFFIX_KEY),
      counts: lex.get(MULTI_SOURCE_COUNTS_LEX_KEY, 0, 0)
    }));
    return Promise.all([initTimeRangeEditor(that), initAlarmTable(that), initCommands(that)]);
  };

  /**
   * Loads a new recipient into the ViewModel.
   *
   * @param {baja.Component} recipient
   * @returns {Promise} Promise resolved when recipient has been loaded
   */
  AlarmConsole.prototype.doLoad = function (recipient) {
    var that = this,
      table = that.$table,
      viewModel = that.$viewModel,
      timeRangeEd = that.$timeRangeEditor,
      tableModel = viewModel.$getTableModel(),
      selection = table.$getSelection(),
      selectedSources = [];
    return recipient.lease().then(function () {
      return Promise.all([timeRangeEd.load(recipient.getDefaultTimeRange()), viewModel.load(recipient)]);
    }).then(function () {
      that.$paginationModel.setRowsPerPage(viewModel.$getPageSize());
      that.$soundOffCmd.update();
      that.$continuousCmd.update();
      return table.load(tableModel);
    }).then(function () {
      var permissions = recipient.getPermissions();
      if (permissions.hasAdminWrite() || permissions.hasAdminInvoke()) {
        that.$forceClearCmd.$showInMenu = true;
      }

      //arm table columns context menu
      new AlarmTableContextMenu(that).arm(that.jq(), SHOW_HIDE_MENU_SELECTOR);

      //arm sorting on row events
      armRowSorting(that);

      //initialize the current sorted column from console options
      that.$initSortColumn();

      // Record the alarm sources currently selected.
      selection.on("changed", function () {
        var rows = tableModel.getRows();
        var selectedElements = selection.getSelectedElements(rows);
        if (!that.isSingleSourceView()) {
          selectedSources = selectedElements.map(function (row) {
            return row.getSubject().source || "";
          });
        }
        if (selectedElements.length > 0) {
          that.$enableMenuItemsForSelectedElements(selectedElements)["catch"](logSevere);
        }
      });

      // After the rows have been reordered then ensure the same sources are selected.
      tableModel.on("rowsReordered", function () {
        if (!that.isSingleSourceView()) {
          tableModel.getRows().map(function (row) {
            return row.getSubject().source || "";
          }).forEach(function (src, i) {
            var index = selectedSources.indexOf(src);
            selection.put(i, index > -1);
          });
          selection.emit('changed');
        }
      });
    });
  };

  /**
   * Destroys the view and view model.
   * @returns {Promise} Promise resolved when view has been destroyed
   */
  AlarmConsole.prototype.doDestroy = function () {
    $(TABLE_CONTAINER_SELECTOR, this.jq()).empty().off();
    return Promise.all([this.getChildWidgets().destroyAll(), this.$viewModel.destroy()]);
  };

  //////////////////////////////////////////////////////////////
  //Alarm Table Context Menu
  //////////////////////////////////////////////////////////////
  ///
  /**
   * @returns {String}
   */
  AlarmConsole.prototype.getContextMenuSelector = function () {
    return 'tr';
  };

  /**
   * @returns {bajaux/commands/CommandGroup}
   */
  AlarmConsole.prototype.updateMenuCommandGroup = function () {
    var cmds = this.getCommandGroup().flatten().filter(function (cmd) {
      return _.result(cmd, 'isShownInMenu');
    });
    return new CommandGroup({
      commands: cmds
    });
  };

  /**
   * @returns {Promise}
   */
  AlarmConsole.prototype.resetAlarmTableSettings = function () {
    var that = this,
      viewModel = that.$viewModel;
    return Promise.all([viewModel.resetAlarmTableSettings(), viewModel.reloadAlarmTableColumns()]).then(function () {
      that.$initSortColumn();
    });
  };

  //////////////////////////////////////////////////////////////
  // Private
  //////////////////////////////////////////////////////////////

  /**
   * @private
   */
  AlarmConsole.prototype.$initSortColumn = function () {
    var view = this,
      viewModel = view.$viewModel,
      singleSrc = view.isSingleSourceView(),
      table = view.$table,
      tableModel = table.getModel(),
      support = viewModel.$getSupport(),
      options = support.getAlarmConsoleOptions(),
      viewOption = singleSrc ? options.singleSourceView : options.multiSourceView,
      sortOption = viewOption.sortByColumn,
      column = tableModel.getColumn(sortOption.column || 'priority'),
      desc = sortOption.desc === undefined ? false : sortOption.desc;
    table.$getThead().children('th').removeClass('asc desc').filter(function () {
      return $(this).data('column') === column;
    }).addClass(desc ? 'desc' : 'asc').data('desc', !desc);
    sortTable(view, column, desc);
  };

  /**
   * @private
   * @returns {module:nmodule/alarm/rc/console/AlarmConsoleViewModel}
   */
  AlarmConsole.prototype.$getViewModel = function () {
    return this.$viewModel;
  };

  /**
   * @private
   * @returns {module:nmodule/bql/rc/fe/DynamicTimeRangeEditor}
   */
  AlarmConsole.prototype.$getTimeRangeEditor = function () {
    return this.$timeRangeEditor;
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/table/Table}
   */
  AlarmConsole.prototype.$getAlarmTable = function () {
    return this.$table;
  };

  /**
   * @private
   * @returns {module:bajaux/util/CommandButtonGroup}
   */
  AlarmConsole.prototype.$getCommandBar = function () {
    return this.jq().find(COMMAND_BAR_SELECTOR).data('widget');
  };

  /**
   * @private
   * @param {Array.<Row>} selectedElements
   * @returns {Promise}
   */
  AlarmConsole.prototype.$enableMenuItemsForSelectedElements = function (selectedElements) {
    var that = this;
    return hasOperatorWrite(selectedElements).then(function (hasRequiredPermissions) {
      that.$forceClearCmd.setEnabled(hasRequiredPermissions);
      that.$ackCmd.setEnabled(hasRequiredPermissions);
      that.$ackRecentCmd.setEnabled(hasRequiredPermissions);
    });
  };

  /**
   * Computes the background color based on the priority and the colors the user has selected
   * @private
   * @param {Number} priority an integer between 0 and 255 that represents the priority of the alarm record
   * @param {module:nmodule/gx/rc/baja/Color} lowColor the low priority color
   * @param {module:nmodule/gx/rc/baja/Color} midColor the mid priority color
   * @param {module:nmodule/gx/rc/baja/Color} highColor the high priority color
   * @returns {module:nmodule/gx/rc/baja/Color}
   */

  AlarmConsole.prototype.$computeBackgroundColor = function (priority, lowColor, midColor, highColor) {
    var mid = 127.5;
    var delta = 127.5;
    var red;
    var blue;
    var green;
    var alpha;
    var startingColor;
    var endingColor;

    // solve for the color using a linear equation y = mx + b,
    // the y axis is the value being monitored, and the x
    // axis is the color (red, green, or blue)
    if (priority > mid) {
      //handle current over boundary
      if (priority > mid + delta) {
        return lowColor;
      }
      startingColor = lowColor;
      endingColor = midColor;
    } else {
      if (priority < mid - delta) {
        return highColor;
      }
      startingColor = midColor;
      endingColor = highColor;
    }
    red = this.$computeNewColorValue(priority, startingColor.getRed(), endingColor.getRed());
    green = this.$computeNewColorValue(priority, startingColor.getGreen(), endingColor.getGreen());
    blue = this.$computeNewColorValue(priority, startingColor.getBlue(), endingColor.getBlue());
    alpha = this.$computeNewColorValue(priority, startingColor.getAlpha(), endingColor.getAlpha());
    return Color.make({
      r: red,
      g: green,
      b: blue,
      a: alpha
    });
  };

  /**
   * Does the math necessary to compute the new color channels values based on the priortiy
   * @private
   * @param {Number} priority
   * @param {Number} start
   * @param {Number} end
   * @returns {Number}
   */
  AlarmConsole.prototype.$computeNewColorValue = function (priority, start, end) {
    var mid = 127.5;
    var delta = 127.5;
    var mColor = (start - end) / delta;
    var bColor = priority > mid ? end - mColor * mid : start - mColor * mid;
    return (mColor * priority + bColor).toFixed();
  };

  //////////////////////////////////////////////////////////////
  // Incubating
  //////////////////////////////////////////////////////////////

  /**
   * Returns true if the view that shows alarms for a single alarm source is being shown.
   *
   * This API is currently incubating and may change in future releases. It's initial use case
   * was for Tridium's Enterprise Security product.
   *
   * @returns {Boolean}
   */
  AlarmConsole.prototype.isSingleSourceView = function () {
    return this.$viewModel.$isSingleSourceModel();
  };

  /**
   * Returns an array the currently selected alarm records.
   *
   * This API is currently incubating and may change in future releases. It's initial use case
   * was for Tridium's Enterprise Security product.
   *
   * @returns {Array.<Object>} An array of the currently selected alarm records.
   */
  AlarmConsole.prototype.getSelectedAlarmRecords = function () {
    var tableModel = this.$viewModel.$getTableModel();
    return this.$table.$getSelection().getSelectedElements(tableModel.getRows()).map(function (row) {
      return row.getSubject();
    });
  };

  /**
   * Returns true if the current table selection is empty.
   *
   * This API is currently incubating and may change in future releases. It's initial use case
   * was for Tridium's Enterprise Security product.
   *
   * @returns {Boolean} true if empty.
   */
  AlarmConsole.prototype.isSelectionEmpty = function () {
    return this.$table.$getSelection().isEmpty();
  };
  return AlarmConsole;
});
