/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/table/AlarmTableContextMenu
 */
define(['underscore', 'Promise', 'lex!alarm', 'dialogs', 'bajaux/commands/Command', 'bajaux/commands/CommandGroup', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/menu/Separator', 'nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu'], function (_, Promise, lex, dialogs, Command, CommandGroup, asyncUtils, feDialogs, Separator, DefaultTableContextMenu) {
  'use strict';

  var alarmLex = lex[0],
    mapAsync = asyncUtils.mapAsync,
    CHECK_ICON = 'module://icons/x16/whiteIcons/check.png';

  /**
   * @class
   * @alias module:nmodule/alarm/rc/console/table/AlarmTableContextMenu
   * @extends module:nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu
   * @param view
   */
  var AlarmTableContextMenu = function AlarmTableContextMenu(view) {
    DefaultTableContextMenu.apply(this, arguments);
    var viewModel = view.$getViewModel(),
      support = viewModel.$getSupport();
    this.$viewModel = viewModel;
    this.$addCommand = new Command({
      displayName: alarmLex.get('alarm.console.commands.addAlarmDataColumn'),
      func: function func() {
        return feDialogs.showFor({
          value: '',
          formFactor: 'mini',
          title: alarmLex.get('alarm.console.commands.addAlarmDataColumn'),
          properties: {
            datalist: viewModel.getAddableColumnNames().join(';')
          }
        }).then(function (name) {
          return viewModel.addAlarmDataColumn(name).then(function () {
            return support.setAlarmConsoleOption('alarmDataColumns', viewModel.getRemovableColumnNames(), viewModel.$isSingleSourceModel());
          });
        });
      }
    });
    this.$removeCommand = new Command({
      displayName: alarmLex.get('alarm.console.commands.removeAlarmDataColumn'),
      func: function func() {
        return feDialogs.showFor({
          value: '',
          formFactor: 'mini',
          title: alarmLex.get('alarm.console.commands.removeAlarmDataColumn'),
          properties: {
            datalist: viewModel.getRemovableColumnNames().join(';')
          }
        }).then(function (name) {
          return viewModel.removeAlarmDataColumn(name).then(function () {
            return support.setAlarmConsoleOption('alarmDataColumns', viewModel.getRemovableColumnNames(), viewModel.$isSingleSourceModel());
          });
        });
      }
    });
    this.$resetCommand = new Command({
      displayName: alarmLex.get('alarm.console.commands.resetTableSettings'),
      func: function func() {
        return dialogs.showYesNo({
          title: alarmLex.get('alarm.console.commands.resetTableSettings'),
          text: viewModel.$isSingleSourceModel() ? alarmLex.get('alarm.console.commands.resetTableSettings.singleSourceView.message') : alarmLex.get('alarm.console.commands.resetTableSettings.multiSourceView.message'),
          yes: function yes() {
            return view.resetAlarmTableSettings();
          }
        });
      }
    });
  };
  AlarmTableContextMenu.prototype = Object.create(DefaultTableContextMenu.prototype);
  AlarmTableContextMenu.prototype.constructor = AlarmTableContextMenu;

  /**
   * @returns {Promise.<module:bajaux/commands/CommandGroup>}
   */
  AlarmTableContextMenu.prototype.toContextMenu = function () {
    var that = this,
      viewModel = that.$viewModel,
      tableModel = viewModel.$getTableModel(),
      columns = _.filter(tableModel.getColumns(), function (col) {
        return col.isHidable();
      });
    return mapAsync(columns, function (column) {
      return Promise.resolve(column.toDisplayName()).then(function (displayName) {
        return new Command({
          displayName: displayName,
          icon: column.isUnseen() ? null : CHECK_ICON,
          func: function func() {
            column.setUnseen(!column.isUnseen());
            return viewModel.setHiddenColumns();
          }
        });
      });
    }).then(function (commands) {
      commands.push(new Separator());
      commands.push(that.$addCommand);
      commands.push(that.$removeCommand);
      commands.push(that.$resetCommand);
      return new CommandGroup({
        commands: commands
      });
    });
  };
  return AlarmTableContextMenu;
});
