/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/db/editors/AlarmTable
 */
define(['baja!alarm:AlarmRecord,bql:DynamicTimeRange,bql:AbsTimeFilter', 'lex!alarm', 'log!nmodule.alarm.rc.db.editors.AlarmTable', 'jquery', 'underscore', 'Promise', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/wb/table/Table', 'nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu', 'nmodule/alarm/rc/db/commands/AlarmDetailsCommand'], function (types, lexs, log, $, _, Promise, fe, Table, DefaultTableContextMenu, AlarmDetailsCommand) {
  "use strict";

  var logSevere = log.severe.bind(log);
  var alarmLex = lexs[0];

  /**
   * @class
   * @alias module:nmodule/alarm/rc/db/editors/AlarmTable
   */
  var AlarmTable = function AlarmTable() {};

  /**
   * Builds the alarm table into the DatabaseMaintenance view.
   *
   * @param {module:nmodule/alarm/rc/db/DatabaseMaintenance} view
   * @param {module:nmodule/webEditors/rc/wb/table/pagination/PaginationModel} paginationModel
   * @returns {Promise}
   */
  AlarmTable.makeFor = function (view, paginationModel) {
    var ALARM_TABLE_SELECTOR = '.alarmTable',
      ALARM_COUNT_LABEL_SELECTOR = '.alarmCount',
      TABLE_CONTAINER_SELECTOR = '.tableContainer',
      COLUMN_FILTER_SELECTOR = '.columnFilter',
      COLUMN_FLAGS_CHANGED_EVENT = 'columnFlagsChanged',
      alarmTable = $(ALARM_TABLE_SELECTOR, view.jq()).data('widget'),
      alarmDetailsCmd;
    function sortTableByColumn(table, column, descending) {
      function getSortColumnObj() {
        var model = table.getModel(),
          th = table.jq().find('thead > th').filter(function () {
            var $this = $(this);
            return $this.hasClass('asc') || $this.hasClass('desc');
          }).first(),
          col = th.data('column');
        col = col || model.getColumn('timestamp');
        return {
          col: col,
          desc: th.data('desc')
        };
      }
      if (!column) {
        var obj = getSortColumnObj();
        column = obj.col;
        descending = obj.desc;
      }
      return Promise.resolve(view.pager && view.pager.update({
        desc: descending,
        col: column
      })).then(function (result) {
        return !result && table.$sortByColumnDisplay(column, descending);
      }).then(function (result) {
        return result ? table : view.table;
      }).then(function (table) {
        var headers = table.$getThead().children('th');
        headers.removeClass('asc desc').data('desc', false);
        var th = headers.filter(function () {
          return $(this).data('column').getName() === column.getName();
        }).first();
        th.addClass(descending ? 'desc' : 'asc').data('desc', descending);
        return table;
      });
    }
    function armEventHandlers(table) {
      armAlarmDetails(table);
      armColumnSorting(table);
      armColumnFilter(table);
      armColumnSizing(table);
      return table;
    }
    function armAlarmDetails(table) {
      //arm double click functionality to launch alarm details dialog
      table.jq().on('dblclick', 'tr', function () {
        if (alarmDetailsCmd) {
          alarmDetailsCmd.invoke();
        } else {
          var filteredGroup = view.getCommandGroup().filter({
            include: function include(cmd) {
              return cmd instanceof AlarmDetailsCommand;
            }
          });
          alarmDetailsCmd = !filteredGroup.isEmpty() && filteredGroup.get(0);
          if (alarmDetailsCmd) {
            alarmDetailsCmd.invoke()["catch"](logSevere);
          }
        }
      });
    }
    function armColumnSorting(table) {
      if (!table.sortByColumn) {
        table.sortByColumn = function (column, descending) {
          return sortTableByColumn(this, column, descending);
        };
      }

      //arm table column sorting
      table.jq().off('click', 'thead > th').on('click', 'thead > th', function () {
        var $this = $(this); //particular th that was clicked
        if ($this.data('column').isSortable()) {
          table.sortByColumn($this.data('column'), !$this.data('desc'));
        }
      });
    }
    function armColumnFilter(table) {
      //add and arm Column filter
      var dom = view.jq();
      $(COLUMN_FILTER_SELECTOR, dom).off(); //turn off existing event handler
      new DefaultTableContextMenu(table).arm(dom, COLUMN_FILTER_SELECTOR);
    }
    function armColumnSizing(table) {
      //set width of last visible column to 100% when column filter is modified
      var model = table.getModel(),
        doLoad = table.doLoad;
      model.on(COLUMN_FLAGS_CHANGED_EVENT, function () {
        sizeVisibleColumns(table);
      });
      table.doLoad = function () {
        return Promise.resolve(doLoad.apply(table, arguments)).then(function () {
          sizeVisibleColumns(table);
        });
      };
    }
    function sizeVisibleColumns(table) {
      function getColumnDom(column) {
        return table.jq().find('.js-col-' + column.getName());
      }
      var model = table.getModel(),
        visibleColumns = _.filter(model.getColumns(), function (col) {
          return !col.isUnseen();
        }),
        length = visibleColumns.length;
      _.each(visibleColumns, function (col, index) {
        getColumnDom(col).css('width', index === length - 1 ? '100%' : 'auto');
      });
      return table;
    }
    function loadNewTable(model) {
      var dom = $('<table class="alarmTable ux-table ux-select-none"></table>');
      return fe.buildFor({
        dom: dom,
        value: model,
        type: Table
      }).then(function (table) {
        dom.appendTo($(TABLE_CONTAINER_SELECTOR, view.jq()));
        alarmTable = table;
      });
    }
    function loadTable() {
      var alarmCount = paginationModel.getRowCount();

      //TODO: incorrect i18n here - should be like alarmLex.get('alarm.alarms', alarmCount)
      $(ALARM_COUNT_LABEL_SELECTOR, view.jq()).text(alarmCount + ' ' + (alarmCount > 1 || alarmCount === 0 ? alarmLex.get('alarm.alarms') : alarmLex.get('alarm.alarm')));
      return Promise.resolve(alarmTable || loadNewTable(paginationModel)).then(function () {
        view.table = alarmTable;
        return alarmTable;
      });
    }
    return loadTable().then(armEventHandlers).then(sizeVisibleColumns);
  };
  return AlarmTable;
});
