/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/fe/AlarmClassEditor
 */
define(['baja!', 'bajaux/events', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/util/spaceUtils', 'css!nmodule/alarm/rc/fe/alarmEditors'], function (baja, events, $, _, fe, BaseEditor, spaceUtils) {
  'use strict';

  var resolveService = spaceUtils.resolveService,
    getAlarmService = _.partial(resolveService, 'alarm:AlarmService'),
    ALARM_BQL_ORD = baja.Ord.make('bql:select name, displayName ' + 'from alarm:AlarmClass'),
    DESTROY_EVENT = events.DESTROY_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT;
  function keyForValue(map, value) {
    return _.find(map.getKeys(), function (key) {
      return map.get(key) === value;
    });
  }

  /**
   * Editor for selecting an alarm class.
   *
   * This editor works by creating a nested String editor with a `datalist`
   * Property. The `datalist` is comprised of the _display_ names of all
   * `AlarmClass`es contained in the `AlarmService`. The user will select a
   * class by display name, and this editor will translate that into the actual
   * slot name of the desired alarm class.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/alarm/rc/fe/AlarmClassEditor
   */
  var AlarmClassEditor = function AlarmClassEditor() {
    BaseEditor.apply(this, arguments);
  };
  AlarmClassEditor.prototype = Object.create(BaseEditor.prototype);
  AlarmClassEditor.prototype.constructor = AlarmClassEditor;

  /**
   * Get the nested string editor holding the class name.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/StringEditor}
   */
  AlarmClassEditor.prototype.$getStringEditor = function () {
    return this.jq().children('.type-baja-String').data('widget');
  };

  /**
   * Get a `baja.OrderedMap` containing the available alarm classes within
   * the `AlarmService` on the station.
   *
   * @returns {Promise}
   */
  AlarmClassEditor.prototype.$getAvailableAlarmClasses = function () {
    var map = new baja.OrderedMap();
    return this.getOrdBase().then(getAlarmService).then(function (alarmService) {
      return ALARM_BQL_ORD.get({
        base: alarmService,
        cursor: {
          each: function each(row) {
            map.put(row.get('displayName'), row.get('name'));
          },
          limit: -1
        }
      });
    }).then(_.constant(map));
  };

  /**
   * Initialize a nested string editor with a `datalist` consisting of the
   * display names of all available alarm classes on the station.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */
  AlarmClassEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on([DESTROY_EVENT, DISABLE_EVENT, ENABLE_EVENT, INITIALIZE_EVENT, LOAD_EVENT, READONLY_EVENT, WRITABLE_EVENT].join(' '), '.editor', false);
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    return that.$getAvailableAlarmClasses().then(function (map) {
      that.$map = map;
    })["catch"](function () {
      that.$map = new baja.OrderedMap();
    }).then(function () {
      return fe.buildFor({
        dom: $('<span/>').appendTo(dom),
        properties: {
          filterWhileTyping: false,
          datalist: that.$map.getKeys().join(';')
        },
        value: '',
        formFactor: 'mini'
      });
    });
  };

  /**
   * Show the display name of the `AlarmClass` corresponding to the loaded
   * slot name. If the loaded slot name does not correspond to an `AlarmClass`
   * on the station, just show the slot name as-is.
   *
   * @param {String} alarmClassName the slot name of the selected alarm class
   * @returns {Promise}
   */
  AlarmClassEditor.prototype.doLoad = function (alarmClassName) {
    var key = keyForValue(this.$map, alarmClassName);
    return this.$getStringEditor().load(key || alarmClassName);
  };

  /**
   * Resolve the slot name of the alarm class corresponding to the display name
   * selected by the user.
   *
   * @returns {*}
   */
  AlarmClassEditor.prototype.doRead = function () {
    var map = this.$map;
    return this.$getStringEditor().read().then(function (displayName) {
      return map.get(displayName) || displayName;
    });
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  AlarmClassEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  AlarmClassEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  AlarmClassEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };
  return AlarmClassEditor;
});
