/*
  This example demonstrates how we can create and use properties in a widget.

  - A property consists of a name and a value.
  - A value has to be either a string, number or boolean value.
  - A property is primarily used as a way to configure the widget externally. 
    For example, from the Px Editor.
  - In a Px environment, a property can be animated and the widget can receive updates.
  - Attributes for a property can be added that give hints on how it should be rendered and persisted.
  - A property for a bajaux Widget is very simple; it's far simpler than a Niagara BComponent property.

  If you're viewing this from a Px page, you'll notice that on the right hand side of the screen the
  'num' property is being animated against a `Ramp` component in the Station. If the Web Widget is not updating
  then please ensure the Ramp is enabled.
*/
define([
  'baja!',
  'baja!baja:Month,baja:Weekday',
  'bajaux/Widget' ], function (
  baja,
  types,
  Widget) {

  'use strict';

  const MAX_LINES = 10;

  /**
   * @class
   * @extends module:bajaux/Widget
   */
  return class PropertiesWidget extends Widget {
    constructor(params) {
      // As of Niagara 4.10, this is the preferred way to set up your widget with default properties.
      // By passing "defaults" to the Widget constructor, your Widget instance will be ready to go
      // with a set of default properties that can then be edited. See the Widget JSDoc for full
      // details.
      // Prior to 4.10, you would call the Widget constructor and then call this.properties().add().

      const defaults = {
        properties: {

          num: 0, // You can just specify a property name and default value.

          // If more than just a name and value is required then you can use an object literal.
          foo: {
            value: true,
            transient: true, // transient: means the value for this property is never persisted.
            hidden: false, // hidden: hides the property in an editor.
            readonly: true // readonly: makes editor for the property readonly in an editor.
          },

          /*
           You can also use Niagara values as Widget properties. When your Widget only runs in
           a browser context, your property values can be anything you like. But when your Widget
           is intended to integrate with Workbench (notably, when embedded into a Px page), then
           Workbench will look for properties on your Widget that have a Niagara Simple type, so
           they can integrate with the rest of the framework. For instance, when your Widget is in a
           Px page, then the Px Editor will look for Simple values in its properties and expose them
           for configuration by the page designer.

           As of Niagara 4.14, there are two ways to configure Niagara values as properties: the
           legacy method, and the new, easy method.

           First, the easy way: simply add a BajaScript value as the property value. The Workbench
           Px Editor will see these properties and expose them for configuration like any other
           Niagara value. DashboardPanes will likewise save these to the DashboardService using the
           correct Type.
           */
          weekday: baja.$('baja:Weekday', 'tuesday'),

          /*
          The legacy way, which prior to Niagara 4.14 was the only way, was to use the string encoding
          of the value, and manually specify the type spec. This method is still fully supported,
          for backwards compatibility, but requires more boilerplate and encoding/decoding of
          strings.
           */
          month: {
            value: 'january',
            typeSpec: 'baja:Month'
          }
        }
      };

      super({ params, defaults });
    }

    doInitialize(dom) {
      dom.html("<pre>Property Changes will appear here...\n</pre>");
    }

    doChanged(name, value) {
      // This will be called every time a property has changed...
      this.$append(`Property changed: name: ${ name } value: ${ value }\n`);
    }

    $append(str) {
      // Append some new text to the widget's 'pre' DOM element...
      const pre = this.jq().find("pre");
      const lines = pre.text().split('\n');
      pre.text(lines.concat(str).slice(-MAX_LINES).join('\n'));
      return this;
    }
  };
});
