/*
Handling events in spandrel widgets
===================================

As of Niagara 4.12, spandrel.jsx can arm event handlers both on DOM elements
and on child widgets.

DOM element event handlers can be armed with a property starting with "on",
followed by the event name (onClick, onChange, etc.). Because bajaux is still
based on jQuery, the argument to the handler will be a jQuery Event and not a
native browser event.

Widget event handlers can be armed with a property starting with "onUx",
followed by the bajaux event name (onUxLoad, onUxModify, etc.). The arguments to
the handler will a jQuery Event, the Widget instance that actually triggered the
event, and any additional arguments it may have passed to the trigger() call.

One near-ubiquitous use case is to handle whenever the user enters a new value
into a child widget. So you don't have to combine an onUxModify event handler
with a read() call every time, spandrel provides a special onUxModifiedValue
handler for convenience. In this case, the read value is the *first* argument,
followed by the jQuery Event and Widget instance.

Point this example at any value. The loaded value will be ignored.
*/

/** @jsx spandrel.jsx */

define([ 'bajaux/spandrel' ], function (spandrel) {

  'use strict';

  const MAX_LOG_LENGTH = 20;

  return class EventLogger extends spandrel((value, { logMessages = [], self }) => {
    return (
      <div>
        <div>
          <button type="button"
                  onClick={(e) => {
                    return self.log('button clicked: event type ' + e.type);
                  }}>
            Click me
          </button>
        </div>

        <div>
          <any value={true} properties={{ trueText: 'Yes', falseText: 'No' }}
               onUxModify={(e, ed) => {
                 return ed.toDisplayName()
                   .then((displayName) => self.log(`received MODIFY_EVENT from ${ displayName }`));
               }}
          />
        </div>

        <div>
          <any value=""
               properties={{ placeholder: 'type here' }}
               onUxModifiedValue={(value, e, ed) => {
                 return ed.toDisplayName()
                   .then((displayName) => self.log(`user typed new value ${ value } in ${ displayName }`));
               }}
          />
        </div>
        <pre>
          <code>{ logMessages.join('\n') }</code>
        </pre>
      </div>
    );
  }, { strategy: 'niagara' }) {

    log(msg) {
      // treat this state() call as magic for now. see details in the "Advanced Widget State" example.
      const { logMessages = [] } = this.state();
      return this.state({ logMessages: [ ...logMessages, msg ].slice(-MAX_LOG_LENGTH) });
    }
  };
});
